/* Copyright (c) 1987, 1988  Stanley T. Shebs. */
/* This program may be used, copied, modified, and redistributed freely */
/* for noncommercial purposes, so long as this notice remains intact. */

/* Unlike most multi-player network programs, xconq does not use multiple */
/* processes.  Instead, it relies on the network capability inherent in  */
/* some graphical interfaces (such as X) to open up multiple displays. */


#include "config.h"
#include "misc.h"
#include "dir.h"
#include "period.h"
#include "side.h"
#include "unit.h"
#include "map.h"
#include "mplay.h"
#include "X11.h"
#include "Motif.h"
#include "global.h"
#include "version.h"
#include "standard.h"

char version[] = VERSION;	/* version string */
char spbuf[BUFSIZE];		/* main printing buffer */
int splen;			/* main printing buffer length */
char tmpbuf[BUFSIZE];		/* a temp buffer */

bool eopt = FALSE;
bool Debug = FALSE;		/* the debugging flag itself. */
bool Build = FALSE;		/* magic flag for scenario-builders */
bool Freeze = FALSE;		/* keeps machine players from moving */
bool Cheat = FALSE;		/* If true allows machine to cheat */

int numfiles = 0;		/* number of data files asked for */
int numgivens = 0;		/* number of sides given on cmd line */
int numhumans = 0;		/* number of bona-fide human players */
int nummaps = 0;		/* Number of mapfiles loaded */

bool givenseen = FALSE;		/* true if world known */
int givenwidth = DEFAULTWIDTH;		/* requested width for a random map */
int givenheight = DEFAULTHEIGHT;	/* requested height for a random map */

Side *winner = NULL;		/* the winning side (allies are also winners) */

char *programname;		/* name of the main program */
char *xconqlib;			/* directory of xconq data files */

Global global;			/* important (i.e. saved/restored) globals */

char *rawfilenames[MAXLOADED];	/* names specified on cmd line */
char *hosts[MAXSIDES];		/* names of displays for each side */

/* Complain and leave if command is garbage. */

void usage_error() {

    fprintf(stderr, "Usage: %s [display] [-A display] [-B] [-C] [-e n] [-m name]\n",
            programname);
    fprintf(stderr, "  [-M w h] [-p name] [-r] [-s name] [-t n] [-T n] [-v]\n");
    exit(1);
}
    
/* List all the specified players briefly. */

void list_players(FILE *fp) {

    int i;

    if (numgivens > 0) {
        fprintf(fp, "%s", (hosts[0] ? hosts[0] : "machine"));
        for (i = 1; i < numgivens; ++i) {
            fprintf(fp, ", %s", (hosts[i] ? hosts[i] : "machine"));
        }
    } else {
        fprintf(fp, "no players defined.");
    }
}

/* Add a player into the array of players, making sure of no overflow. */
/* Fail if so, players probably made a typo on command line - help them */
/* out with a list of what players had been included. */

bool humans[MAXSIDES];	/* flags for which players are human or machine */

void add_player(bool ahuman, char *host) {

    if (numgivens >= MAXSIDES) {
        fprintf(stderr, "At most %d player sides allowed!\n", MAXSIDES);
        fprintf(stderr, "(Valid ones were ");
        list_players(stderr);
        fprintf(stderr, ")\n");
        exit(1);
    }
    hosts[numgivens] = host;
    humans[numgivens] = ahuman;
    numgivens++;
    if (ahuman) numhumans++;
    if (Debug) printf("Added %s player (%s)\n",
                      (ahuman ? "human" : "machine"), (host ? host : ""));
}

void add_default_player() {

    char *default_display = getenv("DISPLAY");

    if (default_display != NULL){
        add_player(TRUE, default_display);
    } else {
        printf("I don't know what display you're on!\n");
        exit(1);
    }
}


/* Shut down displays - should be done before any sort of exit. */

void close_displays() {

    Side *side;

    for_all_sides(side) if (active_display(side)) close_display(side);
}

/* This routine should be called before any sort of non-death exit. */
/* Among other things, it updates the statistics. */

void exit_xconq() {

    int n = 0;
    Unit *unit;
    Side *side;

    close_displays();
    /* Need to kill off all units to finish up statistics */
    for_all_units(side, unit)
      kill_unit(unit, (winner ? VICTOR : SURRENDER));
/*    print_statistics(); */
    /* Offer a one-line comment on the game and then leave */
    for_all_sides(side) if (!side->lost) n++;
    switch (n) {
    case 0:
        printf("\nEverybody lost!\n\n");
        break;
    case 1:
        printf("\nThe %s (%s) won!\n\n",
               plural_form(winner->name),
               (winner->host ? winner->host : "machine"));
        break;
    default:
        sprintf(spbuf, "");
        for_all_sides(side) {
            if (!side->lost) {
                if (strlen(spbuf) > 0) {
                    sprintf(tmpbuf, "/");
                    strcat(spbuf, tmpbuf);
                }
                sprintf(tmpbuf, "%s", side->name);
                strcat(spbuf, tmpbuf);
                if (side->host) {
                    sprintf(tmpbuf, "(%s)", side->host);
                    strcat(spbuf, tmpbuf);
                }
            }
        }
        printf("\nThe %s alliance won!\n\n", spbuf);
        break;
    }
    exit(0);
}

/* Do random small things to get the turn started, including resetting */
/* some side vars that could get things confused, if set wrong. */
/* Most movement-related things should get set later, at beginning of */
/* move phase. */

void init_turn_phase() {

    Side *side;

    ++global.time;
    if (Debug) printf("##### TURN %d #####\n", global.time);
    for_all_sides(side) {        
        show_time(side);
        show_mode(side);
        flush_output(side);
    }
}

/* Quicky test for absence of all units. */

bool all_units_gone(Side *side) {

    int u;

    for_all_unit_types(u) if (side->unitslist[u].units > 0) return FALSE;
    return TRUE;
}

/* Check out all the winning and losing conditions, returning a flag */
/* on which, if any, are true.  The three return values are 1 for win, */
/* -1 for lose, and 0 for undecided. */

int condition_true(Side *side) {

    int i, u, r, amt = 0;
    Unit *unit;
    Condition *cond;

    for (i = 0; i < global.numconds; ++i) {
        cond = &(global.winlose[i]);
        if ((between(cond->starttime, global.time, cond->endtime)) &&
            (cond->sidesn == -1 || cond->sidesn == side_number(side))) {
            switch (cond->type) {
            case TERRITORY:
                for_all_unit_types(u) {
                    amt += side->unitslist[u].units * utypes[u].territory;
                }
                if (cond->win) {
                    if (amt >= cond->n) {
                        notify_all("The %s have enough territory to win!",
                                   plural_form(side->name));
                        return 1;
                    }
                } else {
                    if (amt < cond->n) {
                        notify_all("The %s don't have enough territory!",
                                   plural_form(side->name));
                        return -1;
                    }
                }
                break;
            case UNITCOUNT:
                if (cond->win) {
                    for_all_unit_types(u) {
                        if (side->unitslist[u].units < cond->units[u]) return 0;
                    }
                    notify_all("The %s have enough units to win!",
                               plural_form(side->name));
                    return 1;
                } else {
                    for_all_unit_types(u) {
                        if (side->unitslist[u].units > cond->units[u]) return 0;
                    }
                    notify_all("The %s don't have the required units!",
                               plural_form(side->name));
                    return -1;
                }
                break;
            case RESOURCECOUNT:
                if (cond->win) {
                    for_all_resource_types(r) {
                        if (side->resources[r] < cond->resources[r]) return 0;
                    }
                    notify_all("The %s have enough resources to win!",
                               plural_form(side->name));
                    return 1;
                } else {
                    for_all_resource_types(r) {
                        if (side->resources[r] > cond->resources[r]) return 0;
                    }
                    notify_all("The %s don't have the required resources!",
                               plural_form(side->name));
                    return -1;
                }
                break;
            case POSSESSION:
                unit = unit_at(cond->x, cond->y);
                if (cond->win) {
                    if (unit != NULL && unit->side == side) {
                        notify_all("The %s have got hex %d,%d!",
                                   plural_form(side->name), cond->x, cond->y);
                        return 1;
                    }
                } else {
                    if (unit == NULL || unit->side != side) {
                        notify_all("The %s don't have hex %d,%d!",
                                   plural_form(side->name), cond->x, cond->y);
                        return -1;
                    }
                }
                break;
            default:
                case_panic("condition type", cond->type);
                break;
            }
        }
    }
    return 0;
}

/* Be rude to losers and give all their units to their defeaters (might */
/* not be one, so units become neutral or dead).  Give the loser a chance */
/* to study the screen, mark the side as lost, then pass on this detail to */
/* all the other sides and remove spurious leftover images of units. */

void side_loses(Side *side, Side *victor) {

    int ux, uy;
    Unit *unit;
    Side *side2;

    notify(side, "You lost, sucker");
    if (active_display(side))
        wait_to_exit(side, x_lost);

    side->humanp = FALSE; /* just make sure */
    while (side->unithead->next != side->unithead) {
      unit = side->unithead->next;
      if (alive(unit)) {
        ux = unit->x;  uy = unit->y;
        unit_changes_side(unit, victor, CAPTURE, SURRENDER);
        all_see_hex(ux, uy);
      } else {
        /* units could die instead of changing sides.   Inefficient if */
        /* it happens often, but sides don't resign often.  We need to */
        /* flush the dead units to make sure that the side has no */
        /* units.  Easier this way, since unit_changes_side could get */
        /* a lot of occupants, so we can't just step through the list. */
        flush_dead_units();
      }
    }
    side->lost = TRUE;
    for_all_sides(side2) {
        if (active_display(side2)) {
            remove_images(side2, side_number(side));
            draw_all_sides(side2);
        }
    }
    flush_events();
}

/* Unconditional resignation - usable by everybody. */

void resign_game(Side *side, Side *side2) {

    notify_all("Those wimpy %s have given up!", plural_form(side->name));
    if (side2 != NULL) {
        notify_all("... and they gave all their stuff to the %s!",
                   plural_form(side2->name));
    }
    side_loses(side, side2);
}

/* You're time has run out */

void time_up(Side *side) {

    notify_all("Those slow-pokes %s have run out of time!", plural_form(side->name));
    side_loses(side, NULL);
}

/* Winning is defined as not losing :-) */

void side_wins(Side *side) {

    Side *side2;

    for_all_sides(side2) {
        if (!allied_side(side, side2)) side_loses(side2, (Side *) NULL);
    }
}

/* The win/lose phase assesses the sides' performance during that turn, */
/* and can end the game.  In fact, it is the only way to get out, except */
/* for the quick exit command.  With multiple mixed players, there are a */
/* number of possibilities at game end, such as an all machine win.  Also, */
/* it is possible for all players to lose simultaneously! */

void game_end_phase() {

    bool enemiesleft = FALSE;
    int sidesleft = 0, testrslt;
    Side *side, *side2, *survivor;

    if (Debug) printf("Entering game end phase\n");
    /* See if any sides have won or lost */
    if (global.time >= global.endtime) {
        notify_all("The end of the world has arrived!");
        for_all_sides(side) side_loses(side, (Side *) NULL);
    } else if (Build) {
        /* No winning or losing while building */
    } else {
        for_all_sides(side) {
            if (!side->lost) {
                if (side->timedout) {
                    notify_all("The %s ran out of time!",
                               plural_form(side->name));
                    side_loses(side, (Side *) NULL);
                } else if (all_units_gone(side)) {
                    notify_all("The %s have been wiped out!",
                               plural_form(side->name));
                    side_loses(side, (Side *) NULL);
                } else {
                    testrslt = condition_true(side);
                    if (testrslt > 0) side_wins(side);
                    if (testrslt < 0) side_loses(side, (Side *) NULL);
                }
            }
        }
    }
    /* See if any opposing sides left */
    for_all_sides(side) {
        for_all_sides(side2) {
            if (!side->lost && !side2->lost && enemy_side(side, side2))
                enemiesleft = TRUE;
        }
        if (!side->lost) {
            survivor = side;
            sidesleft++;
        }
    }
    /* Decide if the game is over */
    if (numsides == 1) {
        /* A one-player game can't end */
    } else if (sidesleft == 0) {
        exit_xconq();
    } else if (!enemiesleft) {
        winner = survivor;
        if (sidesleft == 1) {
            notify(winner, "You have prevailed over your enemies!");
        } else {
            for_all_sides(side) {
                if (!side->lost) {
                    notify(side, "Your alliance has defeated its enemies!");
                }
            }
        }
        wait_to_exit(winner, x_exit);
        for_all_sides(side) {
            while (active_display(side))
                process_events();
        }
        exit_xconq();
    } else {
        /* Game isn't over yet */
    }
}

/* The main loop that cycles through the turns and the phases. */
/* Always departs through game end phase (or by core dump :-( ). */

void play() {


    while (TRUE) {
        if (!midturnrestore) {
          init_turn_phase();
          spy_phase();
          disaster_phase();
          build_phase();
          flush_dead_units();
          supply_phase();
          sort_units(FALSE);
        }
        movement_phase();
        consumption_phase();
        flush_dead_units();

        game_end_phase();
    }
}

/* Where it all begins... the main program's primary duty is command line */
/* interpretation, it hands off for everything else. */

main(int argc, char *argv[]) {

    char ch;
    int i, numenemies;

    enter_procedure("Main");
    
    programname = argv[0];
    if ((xconqlib = getenv("XCONQLIB")) == NULL)
        xconqlib = XCONQLIB;
    for (i = 0; i < MAXLOADED; ++i) rawfilenames[i] = "";
    global.giventime = global.timeout = 0;
    add_default_player();

    for (i = 1; i < argc; ++i) {
        if ((argv[i])[0] == '-') {
            ch = (argv[i])[1];
            if (Debug) printf("-%c\n", ch);
            switch (ch) {
            case 'A':
                if (i+1 >= argc) usage_error();
                add_player(FALSE, argv[++i]);
                break;
            case 'B':
                Build = TRUE;
                Freeze = TRUE;
                break;
            case 'C':
                Cheat = TRUE;
                break;
            case 'D':
                Debug++;
                break;
            case 'e':
                if (i+1 >= argc) usage_error();
                eopt = TRUE;
                numenemies = atoi(argv[++i]);
                while (numenemies-- > 0) add_player(FALSE, (char *) NULL);
                break;
            case 'm':
                if (i+1 >= argc) usage_error();
                make_pathname((char *) NULL, argv[++i], "map", spbuf);
                rawfilenames[numfiles++] = copy_string(spbuf);
                break;
            case 'M':
                if (i+2 >= argc) usage_error();
                givenwidth = atoi(argv[++i]);
                givenheight = atoi(argv[++i]);
                break;
            case 'p':
                if (i+1 >= argc) usage_error();
                make_pathname((char *) NULL, argv[++i], "per", spbuf);
                rawfilenames[numfiles++] = copy_string(spbuf);
                break;
            case 'r':
                if (numgivens > 1) {
                    fprintf(stderr, "Warning: -r is resetting the list of\n");
                    fprintf(stderr, "players already given in the command.\n");
                }
                numgivens = numhumans = 0;
                break;
            case 's':
                if (i+1 >= argc) usage_error();
                make_pathname((char *) NULL, argv[++i], "scn", spbuf);
                rawfilenames[numfiles++] = copy_string(spbuf);
                break;
            case 't':
                if (i+1 >= argc) usage_error();
                /* Converting to hunderdths of seconds for internal use */
                global.giventime = (100 * atoi(argv[++i]));
                if (!global.giventime) usage_error();
                break;
            case 'T':
                if (i+1 >= argc) usage_error();
                /* Converting to hunderdths of seconds for internal use */
                global.timeout = (100 * atoi(argv[++i]));
                if (!global.timeout) usage_error();
                break;
            case 'v':
                givenseen = TRUE;
                break;
            default:
                usage_error();
            }
        } else {
            /* We just have a host name for a human player */
            add_player(TRUE, argv[i]);
        }
    }
    /* Put in a single machine opponent if no -e and no human opponents */
    if (!eopt && numgivens == 1) add_player(FALSE, (char *) NULL);
    /* Say hi to the user */
    printf("\n              Welcome to XCONQ version %s\n\n", version);
    maybe_dump_news();


    /* While away the hours setting everything up */
    init_random();
    init_sighandlers();
    init_sides();
    init_units();
    init_game();
    init_Xt();
    init_displays1(argc, argv);
    init_mplayers();

    /* Relatively low chance of screwup now, so safe to delete saved game */
    if (saved_game()) remove_saved_game();

    /* Play! */
    play();
    return(0);
}
