/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.plugin.stream.kafka;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Arrays;
import java.util.Base64;
import java.util.Properties;
import org.apache.commons.lang3.StringUtils;
import org.apache.pinot.shaded.com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaSSLUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(KafkaSSLUtils.class);
    private static final String DEFAULT_CERTIFICATE_TYPE = "X.509";
    private static final String DEFAULT_KEY_ALGORITHM = "RSA";
    private static final String DEFAULT_KEYSTORE_TYPE = "PKCS12";
    private static final String DEFAULT_SECURITY_PROTOCOL = "SSL";
    private static final String DEFAULT_TRUSTSTORE_TYPE = "jks";
    private static final String DEFAULT_SERVER_ALIAS = "ServerAlias";
    private static final String DEFAULT_CLIENT_ALIAS = "ClientAlias";
    private static final String SSL_TRUSTSTORE_LOCATION = "ssl.truststore.location";
    private static final String SSL_TRUSTSTORE_PASSWORD = "ssl.truststore.password";
    private static final String SECURITY_PROTOCOL = "security.protocol";
    private static final String SSL_KEYSTORE_LOCATION = "ssl.keystore.location";
    private static final String SSL_KEYSTORE_PASSWORD = "ssl.keystore.password";
    private static final String SSL_KEY_PASSWORD = "ssl.key.password";
    private static final String STREAM_KAFKA_SSL_SERVER_CERTIFICATE = "stream.kafka.ssl.server.certificate";
    private static final String STREAM_KAFKA_SSL_CERTIFICATE_TYPE = "stream.kafka.ssl.certificate.type";
    private static final String SSL_TRUSTSTORE_TYPE = "ssl.truststore.type";
    private static final String STREAM_KAFKA_SSL_CLIENT_CERTIFICATE = "stream.kafka.ssl.client.certificate";
    private static final String STREAM_KAFKA_SSL_CLIENT_KEY = "stream.kafka.ssl.client.key";
    private static final String STREAM_KAFKA_SSL_CLIENT_KEY_ALGORITHM = "stream.kafka.ssl.client.key.algorithm";
    private static final String SSL_KEYSTORE_TYPE = "ssl.keystore.type";

    private KafkaSSLUtils() {
    }

    public static void initSSL(Properties consumerProps) {
        String trustStoreLocation = consumerProps.getProperty(SSL_TRUSTSTORE_LOCATION);
        String trustStorePassword = consumerProps.getProperty(SSL_TRUSTSTORE_PASSWORD);
        String serverCertificate = consumerProps.getProperty(STREAM_KAFKA_SSL_SERVER_CERTIFICATE);
        if (StringUtils.isAnyEmpty((CharSequence[])new CharSequence[]{trustStoreLocation, trustStorePassword, serverCertificate})) {
            LOGGER.info("Skipping auto SSL server validation since it's not configured.");
            return;
        }
        if (KafkaSSLUtils.shouldRenewTrustStore(consumerProps)) {
            KafkaSSLUtils.initTrustStore(consumerProps);
        }
        String securityProtocol = consumerProps.getProperty(SECURITY_PROTOCOL, DEFAULT_SECURITY_PROTOCOL);
        consumerProps.setProperty(SECURITY_PROTOCOL, securityProtocol);
        String keyStoreLocation = consumerProps.getProperty(SSL_KEYSTORE_LOCATION);
        String keyStorePassword = consumerProps.getProperty(SSL_KEYSTORE_PASSWORD);
        String keyPassword = consumerProps.getProperty(SSL_KEY_PASSWORD);
        String clientCertificate = consumerProps.getProperty(STREAM_KAFKA_SSL_CLIENT_CERTIFICATE);
        if (StringUtils.isAnyEmpty((CharSequence[])new CharSequence[]{keyStoreLocation, keyStorePassword, keyPassword, clientCertificate})) {
            LOGGER.info("Skipping auto SSL client validation since it's not configured.");
            return;
        }
        if (KafkaSSLUtils.shouldRenewKeyStore(consumerProps)) {
            KafkaSSLUtils.initKeyStore(consumerProps);
        }
    }

    @VisibleForTesting
    static void initTrustStore(Properties consumerProps) {
        Path trustStorePath = KafkaSSLUtils.getTrustStorePath(consumerProps);
        if (Files.exists(trustStorePath, new LinkOption[0])) {
            KafkaSSLUtils.deleteFile(trustStorePath);
        }
        LOGGER.info("Initializing the SSL trust store");
        try {
            KafkaSSLUtils.createFile(trustStorePath);
        }
        catch (FileAlreadyExistsException fex) {
            LOGGER.warn("SSL trust store initialization failed as trust store already exists.");
            return;
        }
        catch (IOException iex) {
            throw new RuntimeException(String.format("Failed to create the trust store path: %s", trustStorePath), iex);
        }
        try {
            String trustStorePassword = consumerProps.getProperty(SSL_TRUSTSTORE_PASSWORD);
            String serverCertificate = consumerProps.getProperty(STREAM_KAFKA_SSL_SERVER_CERTIFICATE);
            String certificateType = consumerProps.getProperty(STREAM_KAFKA_SSL_CERTIFICATE_TYPE, DEFAULT_CERTIFICATE_TYPE);
            String trustStoreType = consumerProps.getProperty(SSL_TRUSTSTORE_TYPE, DEFAULT_TRUSTSTORE_TYPE);
            consumerProps.setProperty(SSL_TRUSTSTORE_TYPE, trustStoreType);
            byte[] certBytes = Base64.getDecoder().decode(serverCertificate);
            ByteArrayInputStream certInputStream = new ByteArrayInputStream(certBytes);
            CertificateFactory certificateFactory = CertificateFactory.getInstance(certificateType);
            Certificate certificate = certificateFactory.generateCertificate(certInputStream);
            KeyStore trustStore = KeyStore.getInstance(trustStoreType);
            trustStore.load(null, null);
            trustStore.setCertificateEntry(DEFAULT_SERVER_ALIAS, certificate);
            try (FileOutputStream fos = new FileOutputStream(trustStorePath.toString());){
                trustStore.store(fos, trustStorePassword.toCharArray());
            }
            LOGGER.info("Initialized the SSL trust store.");
        }
        catch (Exception ex) {
            throw new RuntimeException("Error initializing the SSL trust store", ex);
        }
    }

    @VisibleForTesting
    static void initKeyStore(Properties consumerProps) {
        Path keyStorePath = KafkaSSLUtils.getKeyStorePath(consumerProps);
        if (Files.exists(keyStorePath, new LinkOption[0])) {
            KafkaSSLUtils.deleteFile(keyStorePath);
        }
        LOGGER.info("Initializing the SSL key store");
        try {
            KafkaSSLUtils.createFile(keyStorePath);
        }
        catch (FileAlreadyExistsException fex) {
            LOGGER.warn("SSL key store initialization failed as key store already exists.");
            return;
        }
        catch (IOException iex) {
            throw new RuntimeException(String.format("Failed to create the key store path: %s", keyStorePath), iex);
        }
        String keyStorePassword = consumerProps.getProperty(SSL_KEYSTORE_PASSWORD);
        String keyPassword = consumerProps.getProperty(SSL_KEY_PASSWORD);
        String clientCertificate = consumerProps.getProperty(STREAM_KAFKA_SSL_CLIENT_CERTIFICATE);
        String certificateType = consumerProps.getProperty(STREAM_KAFKA_SSL_CERTIFICATE_TYPE, DEFAULT_CERTIFICATE_TYPE);
        String privateKeyString = consumerProps.getProperty(STREAM_KAFKA_SSL_CLIENT_KEY);
        String privateKeyAlgorithm = consumerProps.getProperty(STREAM_KAFKA_SSL_CLIENT_KEY_ALGORITHM, DEFAULT_KEY_ALGORITHM);
        String keyStoreType = consumerProps.getProperty(SSL_KEYSTORE_TYPE, DEFAULT_KEYSTORE_TYPE);
        consumerProps.setProperty(SSL_KEYSTORE_TYPE, keyStoreType);
        try {
            byte[] pkBytes = Base64.getDecoder().decode(privateKeyString);
            byte[] certBytes = Base64.getDecoder().decode(clientCertificate);
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(pkBytes);
            KeyFactory keyFactory = KeyFactory.getInstance(privateKeyAlgorithm);
            PrivateKey privateKey = keyFactory.generatePrivate(keySpec);
            CertificateFactory certFactory = CertificateFactory.getInstance(certificateType);
            ByteArrayInputStream certInputStream = new ByteArrayInputStream(certBytes);
            Certificate certificate = certFactory.generateCertificate(certInputStream);
            KeyStore keyStore = KeyStore.getInstance(keyStoreType);
            keyStore.load(null, null);
            KeyStore.PrivateKeyEntry privateKeyEntry = new KeyStore.PrivateKeyEntry(privateKey, new Certificate[]{certificate});
            KeyStore.PasswordProtection keyPasswordProtection = new KeyStore.PasswordProtection(keyPassword.toCharArray());
            keyStore.setEntry(DEFAULT_CLIENT_ALIAS, privateKeyEntry, keyPasswordProtection);
            try (FileOutputStream fos = new FileOutputStream(keyStorePath.toString());){
                keyStore.store(fos, keyStorePassword.toCharArray());
            }
            LOGGER.info("Initialized the SSL key store.");
        }
        catch (Exception ex) {
            throw new RuntimeException("Error initializing the SSL key store", ex);
        }
    }

    private static Path getTrustStorePath(Properties consumerProps) {
        String trustStoreLocation = consumerProps.getProperty(SSL_TRUSTSTORE_LOCATION);
        return Paths.get(trustStoreLocation, new String[0]);
    }

    private static Path getKeyStorePath(Properties consumerProps) {
        String keyStoreLocation = consumerProps.getProperty(SSL_KEYSTORE_LOCATION);
        return Paths.get(keyStoreLocation, new String[0]);
    }

    private static boolean shouldRenewTrustStore(Properties consumerProps) {
        boolean renewTrustStore;
        Path trustStorePath = KafkaSSLUtils.getTrustStorePath(consumerProps);
        String trustStorePassword = consumerProps.getProperty(SSL_TRUSTSTORE_PASSWORD);
        String serverCertificate = consumerProps.getProperty(STREAM_KAFKA_SSL_SERVER_CERTIFICATE);
        String certificateType = consumerProps.getProperty(STREAM_KAFKA_SSL_CERTIFICATE_TYPE, DEFAULT_CERTIFICATE_TYPE);
        try {
            KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
            try (FileInputStream fis = new FileInputStream(trustStorePath.toString());){
                trustStore.load(fis, trustStorePassword.toCharArray());
            }
            byte[] decodedCertBytes = Base64.getDecoder().decode(serverCertificate);
            CertificateFactory certFactory = CertificateFactory.getInstance(certificateType);
            Certificate providedCertificate = certFactory.generateCertificate(new ByteArrayInputStream(decodedCertBytes));
            Certificate trustStoreCertificate = trustStore.getCertificate(DEFAULT_SERVER_ALIAS);
            renewTrustStore = !providedCertificate.equals(trustStoreCertificate);
        }
        catch (FileNotFoundException fex) {
            renewTrustStore = true;
        }
        catch (Exception ex) {
            renewTrustStore = true;
            LOGGER.warn("Trust store certificate comparison check failed.", ex);
        }
        return renewTrustStore;
    }

    private static boolean shouldRenewKeyStore(Properties consumerProps) {
        boolean renewKeyStore;
        Path keyStorePath = KafkaSSLUtils.getKeyStorePath(consumerProps);
        String keyStorePassword = consumerProps.getProperty(SSL_KEYSTORE_PASSWORD);
        String keyPassword = consumerProps.getProperty(SSL_KEY_PASSWORD);
        String certificateType = consumerProps.getProperty(STREAM_KAFKA_SSL_CERTIFICATE_TYPE, DEFAULT_CERTIFICATE_TYPE);
        String clientCertificate = consumerProps.getProperty(STREAM_KAFKA_SSL_CLIENT_CERTIFICATE);
        String privateKeyAlgorithm = consumerProps.getProperty(STREAM_KAFKA_SSL_CLIENT_KEY_ALGORITHM, DEFAULT_KEY_ALGORITHM);
        String privateKeyString = consumerProps.getProperty(STREAM_KAFKA_SSL_CLIENT_KEY);
        try {
            KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            try (FileInputStream fis = new FileInputStream(keyStorePath.toString());){
                keyStore.load(fis, keyStorePassword.toCharArray());
            }
            Certificate keyStoreCert = keyStore.getCertificate(DEFAULT_CLIENT_ALIAS);
            PrivateKey keyStorePrivateKey = (PrivateKey)keyStore.getKey(DEFAULT_CLIENT_ALIAS, keyPassword.toCharArray());
            CertificateFactory certFactory = CertificateFactory.getInstance(certificateType);
            Certificate providedCert = certFactory.generateCertificate(new ByteArrayInputStream(Base64.getDecoder().decode(clientCertificate)));
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(Base64.getDecoder().decode(privateKeyString));
            KeyFactory keyFactory = KeyFactory.getInstance(privateKeyAlgorithm);
            PrivateKey providedPrivateKey = keyFactory.generatePrivate(keySpec);
            boolean isCertSame = Arrays.equals(keyStoreCert.getEncoded(), providedCert.getEncoded());
            boolean isKeySame = Arrays.equals(keyStorePrivateKey.getEncoded(), providedPrivateKey.getEncoded());
            renewKeyStore = !isCertSame || !isKeySame;
        }
        catch (FileNotFoundException fex) {
            renewKeyStore = true;
        }
        catch (Exception ex) {
            renewKeyStore = true;
            LOGGER.warn("Key store certificate and private key comparison checks failed.", ex);
        }
        return renewKeyStore;
    }

    private static void deleteFile(Path path) {
        try {
            Files.deleteIfExists(path);
            LOGGER.info(String.format("Successfully deleted file: %s", path));
        }
        catch (IOException iex) {
            LOGGER.warn(String.format("Failed to delete the file: %s", path));
        }
    }

    private static void createFile(Path path) throws IOException {
        Path filePath;
        Path parentDir = path.getParent();
        if (parentDir != null) {
            Files.createDirectories(parentDir, new FileAttribute[0]);
        }
        if (!Files.exists(filePath = path.toAbsolutePath(), new LinkOption[0])) {
            Files.createFile(filePath, new FileAttribute[0]);
            LOGGER.info(String.format("Successfully created file: %s", path));
        }
    }
}

