/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.core;

import java.util.Calendar;
import java.util.HashSet;
import java.util.TimeZone;
import javax.jcr.ItemNotFoundException;
import javax.jcr.RepositoryException;
import org.apache.jackrabbit.core.RepositoryImpl;
import org.apache.jackrabbit.core.id.NodeId;
import org.apache.jackrabbit.core.id.PropertyId;
import org.apache.jackrabbit.core.persistence.PersistenceManager;
import org.apache.jackrabbit.core.state.ChangeLog;
import org.apache.jackrabbit.core.state.ChildNodeEntry;
import org.apache.jackrabbit.core.state.ItemStateException;
import org.apache.jackrabbit.core.state.NodeState;
import org.apache.jackrabbit.core.version.InconsistentVersioningState;
import org.apache.jackrabbit.core.version.InternalVersion;
import org.apache.jackrabbit.core.version.InternalVersionHistory;
import org.apache.jackrabbit.core.version.InternalVersionManagerImpl;
import org.apache.jackrabbit.core.version.VersionHistoryInfo;
import org.apache.jackrabbit.spi.Name;
import org.apache.jackrabbit.spi.NameFactory;
import org.apache.jackrabbit.spi.commons.name.NameConstants;
import org.apache.jackrabbit.spi.commons.name.NameFactoryImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RepositoryChecker {
    private static final Logger log = LoggerFactory.getLogger(RepositoryChecker.class);
    private final PersistenceManager workspace;
    private final ChangeLog workspaceChanges;
    private final ChangeLog vworkspaceChanges;
    private final InternalVersionManagerImpl versionManager;
    private static final long CHUNKSIZE = 256L;
    private long dirtyNodes = 0L;
    private long totalNodes = 0L;
    private long brokenNodes = 0L;
    private long startTime;

    public RepositoryChecker(PersistenceManager workspace, InternalVersionManagerImpl versionManager) {
        this.workspace = workspace;
        this.workspaceChanges = new ChangeLog();
        this.vworkspaceChanges = new ChangeLog();
        this.versionManager = versionManager;
    }

    public void check(NodeId id, boolean recurse, boolean fixImmediately) throws RepositoryException {
        log.info("Starting RepositoryChecker");
        this.startTime = System.currentTimeMillis();
        this.internalCheck(id, recurse, fixImmediately);
        if (fixImmediately) {
            this.internalFix(true);
        }
        log.info("RepositoryChecker finished; checked " + this.totalNodes + " nodes in " + (System.currentTimeMillis() - this.startTime) + "ms, problems found: " + this.brokenNodes);
    }

    private void internalCheck(NodeId id, boolean recurse, boolean fixImmediately) throws RepositoryException {
        try {
            log.debug("Checking consistency of node {}", (Object)id);
            ++this.totalNodes;
            NodeState state = this.workspace.load(id);
            this.checkVersionHistory(state);
            if (fixImmediately && this.dirtyNodes > 256L) {
                this.internalFix(false);
            }
            if (recurse) {
                for (ChildNodeEntry child : state.getChildNodeEntries()) {
                    if (RepositoryImpl.SYSTEM_ROOT_NODE_ID.equals(child.getId())) continue;
                    this.internalCheck(child.getId(), recurse, fixImmediately);
                }
            }
        }
        catch (ItemStateException e) {
            throw new RepositoryException("Unable to access node " + String.valueOf(id), e);
        }
    }

    private void fix(PersistenceManager pm, ChangeLog changes, String store, boolean verbose) throws RepositoryException {
        if (changes.hasUpdates()) {
            if (log.isWarnEnabled()) {
                log.warn("Fixing " + store + " inconsistencies: " + changes.toString());
            }
            try {
                pm.store(changes);
                changes.reset();
            }
            catch (ItemStateException e) {
                String message = "Failed to fix " + store + " inconsistencies (aborting)";
                log.error(message, e);
                throw new RepositoryException(message, e);
            }
        } else if (verbose) {
            log.info("No " + store + " inconsistencies found");
        }
    }

    public void fix() throws RepositoryException {
        this.internalFix(true);
    }

    private void internalFix(boolean verbose) throws RepositoryException {
        this.fix(this.workspace, this.workspaceChanges, "workspace", verbose);
        this.fix(this.versionManager.getPersistenceManager(), this.vworkspaceChanges, "versioning workspace", verbose);
        this.dirtyNodes = 0L;
    }

    private void checkVersionHistory(NodeState node) {
        String message = null;
        NodeId nid = node.getNodeId();
        boolean isVersioned = node.hasPropertyName(NameConstants.JCR_VERSIONHISTORY);
        NodeId vhid = null;
        try {
            InternalVersionHistory vh;
            String intro;
            block13: {
                String type = isVersioned ? "in-use" : "candidate";
                log.debug("Checking " + type + " version history of node {}", (Object)nid);
                intro = "Removing references to an inconsistent " + type + " version history of node " + String.valueOf(nid);
                message = intro + " (getting the VersionInfo)";
                VersionHistoryInfo vhi = this.versionManager.getVersionHistoryInfoForNode(node);
                if (vhi != null) {
                    vhid = vhi.getVersionHistoryId();
                }
                message = intro + " (getting the InternalVersionHistory)";
                vh = null;
                try {
                    vh = this.versionManager.getVersionHistoryOfNode(nid);
                }
                catch (ItemNotFoundException ex) {
                    if (!isVersioned) break block13;
                    throw ex;
                }
            }
            if (vh == null) {
                if (isVersioned) {
                    message = intro + "getVersionHistoryOfNode returned null";
                    throw new InconsistentVersioningState(message);
                }
            } else {
                vhid = vh.getId();
                message = intro + " (getting the version names failed)";
                Name[] versionNames = vh.getVersionNames();
                boolean seenRoot = false;
                for (Name versionName : versionNames) {
                    seenRoot |= NameConstants.JCR_ROOTVERSION.equals(versionName);
                    log.debug("Checking version history of node {}, version {}", (Object)nid, (Object)versionName);
                    message = intro + " (getting version " + String.valueOf(versionName) + "  failed)";
                    InternalVersion v = vh.getVersion(versionName);
                    message = intro + "(frozen node of root version " + String.valueOf(v.getId()) + " missing)";
                    if (null != v.getFrozenNode()) continue;
                    throw new InconsistentVersioningState(message);
                }
                if (!seenRoot) {
                    message = intro + " (root version is missing)";
                    throw new InconsistentVersioningState(message);
                }
            }
        }
        catch (InconsistentVersioningState e) {
            log.info(message, e);
            NodeId nvhid = e.getVersionHistoryNodeId();
            if (nvhid != null) {
                if (vhid != null && !nvhid.equals(vhid)) {
                    log.error("vhrid returned with InconsistentVersioningState does not match the id we already had: " + String.valueOf(vhid) + " vs " + String.valueOf(nvhid));
                }
                vhid = nvhid;
            }
            this.removeVersionHistoryReferences(node, vhid);
        }
        catch (Exception e) {
            log.info(message, e);
            this.removeVersionHistoryReferences(node, vhid);
        }
    }

    private void removeVersionHistoryReferences(NodeState node, NodeId vhid) {
        ++this.dirtyNodes;
        ++this.brokenNodes;
        NodeState modified = new NodeState(node, 2, true);
        HashSet<Name> mixins = new HashSet<Name>(node.getMixinTypeNames());
        if (mixins.remove(NameConstants.MIX_VERSIONABLE)) {
            mixins.add(NameConstants.MIX_REFERENCEABLE);
            modified.setMixinTypeNames(mixins);
        }
        this.removeProperty(modified, NameConstants.JCR_VERSIONHISTORY);
        this.removeProperty(modified, NameConstants.JCR_BASEVERSION);
        this.removeProperty(modified, NameConstants.JCR_PREDECESSORS);
        this.removeProperty(modified, NameConstants.JCR_ISCHECKEDOUT);
        this.workspaceChanges.modified(modified);
        if (vhid != null) {
            log.info("trying to rename version history of node " + String.valueOf(node.getId()));
            NameFactory nf = NameFactoryImpl.getInstance();
            Name vhrname = nf.create("", node.getId().toString());
            try {
                NodeState vhrState = this.versionManager.getPersistenceManager().load(vhid);
                NodeState vhrParentState = this.versionManager.getPersistenceManager().load(vhrState.getParentId());
                if (vhrParentState.hasChildNodeEntry(vhrname)) {
                    NodeState modifiedParent = (NodeState)this.vworkspaceChanges.get(vhrState.getParentId());
                    if (modifiedParent == null) {
                        modifiedParent = new NodeState(vhrParentState, 2, true);
                    }
                    Calendar now = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
                    String appendme = String.format(" (disconnected by RepositoryChecker on %04d%02d%02dT%02d%02d%02dZ)", now.get(1), now.get(2) + 1, now.get(5), now.get(11), now.get(12), now.get(13));
                    modifiedParent.renameChildNodeEntry(vhid, nf.create(vhrname.getNamespaceURI(), vhrname.getLocalName() + appendme));
                    this.vworkspaceChanges.modified(modifiedParent);
                } else {
                    log.info("child node entry " + String.valueOf(vhrname) + " for version history not found inside parent folder.");
                }
            }
            catch (Exception ex) {
                log.error("while trying to rename the version history", ex);
            }
        }
    }

    private void removeProperty(NodeState node, Name name) {
        if (node.hasPropertyName(name)) {
            node.removePropertyName(name);
            try {
                this.workspaceChanges.deleted(this.workspace.load(new PropertyId(node.getNodeId(), name)));
            }
            catch (ItemStateException itemStateException) {
                // empty catch block
            }
        }
    }
}

