################################################################################
#  Licensed to the Apache Software Foundation (ASF) under one
#  or more contributor license agreements.  See the NOTICE file
#  distributed with this work for additional information
#  regarding copyright ownership.  The ASF licenses this file
#  to you under the Apache License, Version 2.0 (the
#  "License"); you may not use this file except in compliance
#  with the License.  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
# limitations under the License.
#################################################################################
import sys
from pathlib import Path

from flink_agents.plan.agent_plan import AgentPlan
from flink_agents.plan.function import JavaFunction

# The java agent plan json to be checked is generated by
# flink-agents/plan/src/test/java/org/apache/flink/agents/plan/
# compatibility/GenerateAgentPlanJson.java
if __name__ == "__main__":
    json_path = sys.argv[1]
    with Path.open(Path(json_path)) as f:
        java_plan_json = f.read()

    agent_plan = AgentPlan.model_validate_json(java_plan_json)
    actions = agent_plan.actions

    assert len(actions) == 4

    event = "org.apache.flink.agents.api.Event"
    input_event = "org.apache.flink.agents.api.InputEvent"
    my_event = (
        "org.apache.flink.agents.plan.compatibility.GenerateAgentPlanJson$MyEvent"
    )
    runner_context = "org.apache.flink.agents.api.context.RunnerContext"
    qualname = "org.apache.flink.agents.plan.compatibility.GenerateAgentPlanJson$JavaAgentPlanCompatibilityTestAgent"

    # check the first action
    assert "firstAction" in actions
    action1 = actions["firstAction"]
    func1 = action1.exec
    assert isinstance(func1, JavaFunction)
    assert func1.qualname == qualname
    assert func1.method_name == "firstAction"
    assert func1.parameter_types == [input_event, runner_context]
    listen_event_types1 = action1.listen_event_types
    assert listen_event_types1 == [input_event]

    # check the second action
    assert "secondAction" in actions
    action2 = actions["secondAction"]
    func2 = action2.exec
    assert isinstance(func2, JavaFunction)
    assert func2.qualname == qualname
    assert func2.method_name == "secondAction"
    assert func2.parameter_types == [
        event,
        runner_context,
    ]
    listen_event_types2 = action2.listen_event_types
    assert sorted(listen_event_types2) == [
        input_event,
        my_event,
    ]

    # check actions_by_event
    actions_by_event = agent_plan.actions_by_event
    assert len(actions_by_event) == 5

    assert input_event in actions_by_event
    assert sorted(actions_by_event[input_event]) == ["firstAction", "secondAction"]

    assert my_event in actions_by_event
    assert actions_by_event[my_event] == ["secondAction"]
