#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
from __future__ import annotations

from dataclasses import dataclass, field
from functools import cache
from typing import Any

from airflow.listeners import hookimpl
from airflow.utils.state import TaskInstanceState


@dataclass
class ListenerState:
    started_component: Any = None
    stopped_component: Any = None
    state: list[Any] = field(default_factory=list)


@cache
def get_listener_state() -> ListenerState:
    return ListenerState()


@hookimpl
def on_starting(component):
    get_listener_state().started_component = component


@hookimpl
def before_stopping(component):
    get_listener_state().stopped_component = component


@hookimpl
def on_task_instance_running(previous_state, task_instance):
    get_listener_state().state.append(TaskInstanceState.RUNNING)


@hookimpl
def on_task_instance_success(previous_state, task_instance):
    get_listener_state().state.append(TaskInstanceState.SUCCESS)


@hookimpl
def on_task_instance_failed(previous_state, task_instance, error: None | str | BaseException):
    get_listener_state().state.append(TaskInstanceState.FAILED)


def clear():
    listener_state = get_listener_state()
    listener_state.started_component = None
    listener_state.stopped_component = None
    listener_state.state = []
