##########################################################################
##
#O DeclearOperations( "Poset", [ _P, _rel] )
##
## Returns the poset defined on the points  <_P>  and the relations 
## generated by  <_rel>.  The elements in  <_P>  is given as a list, for
## example ["a", "b", "c", "d"] and the relations are given as a list of 
## lists, for instance in the above case:  [ ["a", "b", "c"], ["b", "d"], 
## ["c", "d"]]. The first list means that "a" < "b" and "a" < "c",  and
## the second one means "b" < "d"  and finally the last one means 
## "c" < "d". 
##
InstallMethod( Poset,
    "for sets",
    [ IsList, IsList ],
        
    function( _P, _rel )
    local points, D, rel, r, i, P, poset, min_rels, lessthan;
    
    if Length( _P ) = 0 then
        Error( "the first argument is an empty set,\n" );
    fi;
    #
    # Checking if the relations are given in terms of elements of the underlying
    # set given in the argument  <_P>. 
    #
    points := Union( List([1..Length(_rel)], x -> Set( _rel[x] ) ) );
    if not IsSubsetSet( Set( _P ), points ) then
        Error( Difference( points, Set( _P ) ), " occur in the relations but not in the underlying set for the poset,\n");
    fi;
    #
    # Defining the generating relations in the direct product <_P> x <_P>.
    #
    D := Domain( _P );
    _P := Elements( D );
    rel := [];
    for r in _rel do
        for i in [ 2..Length( r ) ] do
            Add(rel, DirectProductElement( [ Elements( D )[Position( _P, r[ 1 ] ) ], Elements( D )[ Position( _P, r[ i ] ) ] ] ) );
        od;
    od;
    #
    # Forming the binary relation generated by these relations, closing up under reflexitivity and transitivity
    # and checking if it forms a partially ordered set.  If so, this data structure is returned. 
    # 
    P := BinaryRelationByElements( D, rel );
    P := ReflexiveClosureBinaryRelation( P );
    P := TransitiveClosureBinaryRelation( P );
    if IsPartialOrderBinaryRelation( P ) then
        min_rels := AsList( UnderlyingRelation( HasseDiagramBinaryRelation( P ) ) );
        lessthan := function( x, y ); 
            if ( x in D ) and ( y in D ) then 
                if DirectProductElement( [ x, y ] ) in Elements( UnderlyingRelation( P ) ) then 
                    return true;
                else
                    return false;
                fi;
            else
                Error( "entered elements are not in the underlying set of the poset,\n" );
            fi;
        end;
        
        poset := rec( set := Elements( Source( P ) ), binary_relation := P, 
                      minimal_relations := min_rels, lessthan := lessthan );
        ObjectifyWithAttributes( poset, TheTypePoset );
        return poset;
    else
        Error("entered relations doesn't define a partial order,\n");
    fi;
end
  );

InstallMethod( Size,
    "for posets",
    [ IsPoset ],
        
    function( P )
    
    return Size( P!.set );
end 
  );

InstallMethod( UnderlyingSet,
    "for posets",
    [ IsPoset ],
        
    function( P )
    
    return P!.set;
end 
  );

InstallMethod( PartialOrderOfPoset,
    "for posets",
    [ IsPoset ],
        
    function( P )
    
    return P!.lessthan;
end 
  );

InstallMethod( ViewObj,
    "for posets",
    [ IsPoset ],
        
    function( P )
    
    Print( "<A poset on ", Size( P ), " points>" );
end 
  );

InstallMethod( Display,
    "for posets",
    [ IsPoset ],
        
    function( P )
    
    Print( "A poset with underlying set:\n", UnderlyingSet( P ), "\nand partial order:\n", P!.minimal_relations, "\n" );
end 
  );