package proxy

import (
	"io"
	"net"
	"os"
	"path/filepath"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"gitlab.com/gitlab-org/step-runner/pkg/api/internal/test"
)

func Test_Proxy(t *testing.T) {
	// Create a unique socket path for this test to avoid conflicts
	tmpDir := t.TempDir()
	socketPath := filepath.Join(tmpDir, "test-step-runner.sock")

	// Remove any existing socket file
	_ = os.Remove(socketPath)

	ln, err := net.ListenUnix("unix", &net.UnixAddr{Name: socketPath, Net: "unix"})
	require.NoError(t, err)
	defer func() {
		assert.NoError(t, ln.Close())
		_ = os.Remove(socketPath)
	}()

	var conn net.Conn
	go func() {
		conn, err = ln.Accept()
		require.NoError(t, err)
		_, err = io.Copy(conn, conn)
		assert.NoError(t, err)
	}()

	// use a pipe for the pr and pw to pass to the proxy
	pr, pw := io.Pipe()
	buf := test.SyncBuff{}

	// start the proxy...
	go func() {
		conn, err := net.DialUnix("unix", nil, &net.UnixAddr{Name: socketPath, Net: "unix"})
		require.NoError(t, err)
		assert.NoError(t, Proxy(pr, &buf, conn))
	}()

	// write some stuff into the writer 1/2 of the pipe...
	for _, p := range [][]byte{
		[]byte("aaaa"),
		[]byte("bbbb"),
		[]byte("cccc"),
	} {
		_, err := pw.Write(p)
		assert.NoError(t, err)
	}
	// wait for the above stuff to be written into the receiving buffer...
	assert.Eventually(t, func() bool { return buf.String() == "aaaabbbbcccc" }, time.Second, time.Millisecond*100)
	// Close the PipeWriter; this will exit the reading loop in the proxy
	assert.NoError(t, pw.Close())
}
