# sed

The `config_*.h` and `config_*.bzl` files were generated by running `./configure` on the relevant
platform and invoking the following script:

```python
"""A script for parsing data from `configure` outputs of `sed`."""

import json
import platform
import re
import shutil
from pathlib import Path

# Note that some values may be missing for platforms not currently tested.
KNOWN_VALUES = {
    "GNULIB__EXIT",
    "GNULIB_ACCESS",
    "GNULIB_ALIGNED_ALLOC",
    "GNULIB_ATOLL",
    "GNULIB_BTOWC",
    "GNULIB_CALLOC_GNU",
    "GNULIB_CALLOC_POSIX",
    "GNULIB_CANONICALIZE_FILE_NAME",
    "GNULIB_CHDIR",
    "GNULIB_CHMOD",
    "GNULIB_CHOWN",
    "GNULIB_CLOSE",
    "GNULIB_COPY_FILE_RANGE",
    "GNULIB_CREAT",
    "GNULIB_CTIME",
    "GNULIB_DPRINTF",
    "GNULIB_DUP",
    "GNULIB_DUP2",
    "GNULIB_DUP3",
    "GNULIB_DUPLOCALE",
    "GNULIB_ENVIRON",
    "GNULIB_EUIDACCESS",
    "GNULIB_EXECL",
    "GNULIB_EXECLE",
    "GNULIB_EXECLP",
    "GNULIB_EXECV",
    "GNULIB_EXECVE",
    "GNULIB_EXECVP",
    "GNULIB_EXECVPE",
    "GNULIB_EXPLICIT_BZERO",
    "GNULIB_FACCESSAT",
    "GNULIB_FCHDIR",
    "GNULIB_FCHMODAT",
    "GNULIB_FCHOWNAT",
    "GNULIB_FCLOSE",
    "GNULIB_FCNTL",
    "GNULIB_FDATASYNC",
    "GNULIB_FDOPEN",
    "GNULIB_FFLUSH",
    "GNULIB_FFSL",
    "GNULIB_FFSLL",
    "GNULIB_FGETC",
    "GNULIB_FGETS",
    "GNULIB_FOPEN",
    "GNULIB_FPRINTF_POSIX",
    "GNULIB_FPRINTF",
    "GNULIB_FPURGE",
    "GNULIB_FPUTC",
    "GNULIB_FPUTS",
    "GNULIB_FREAD",
    "GNULIB_FREE_POSIX",
    "GNULIB_FREOPEN",
    "GNULIB_FSCANF",
    "GNULIB_FSEEK",
    "GNULIB_FSEEKO",
    "GNULIB_FSTAT",
    "GNULIB_FSTATAT",
    "GNULIB_FSYNC",
    "GNULIB_FTELL",
    "GNULIB_FTELLO",
    "GNULIB_FTRUNCATE",
    "GNULIB_FUTIMENS",
    "GNULIB_FWRITE",
    "GNULIB_GETC",
    "GNULIB_GETCHAR",
    "GNULIB_GETCWD",
    "GNULIB_GETDELIM",
    "GNULIB_GETDOMAINNAME",
    "GNULIB_GETDTABLESIZE",
    "GNULIB_GETENTROPY",
    "GNULIB_GETGROUPS",
    "GNULIB_GETHOSTNAME",
    "GNULIB_GETLINE",
    "GNULIB_GETLOADAVG",
    "GNULIB_GETLOGIN_R",
    "GNULIB_GETLOGIN",
    "GNULIB_GETOPT_POSIX",
    "GNULIB_GETPAGESIZE",
    "GNULIB_GETPASS",
    "GNULIB_GETRANDOM",
    "GNULIB_GETSUBOPT",
    "GNULIB_GETUMASK",
    "GNULIB_GETUSERSHELL",
    "GNULIB_GRANTPT",
    "GNULIB_GROUP_MEMBER",
    "GNULIB_ISATTY",
    "GNULIB_ISBLANK",
    "GNULIB_ISWBLANK",
    "GNULIB_ISWCTYPE",
    "GNULIB_ISWDIGIT",
    "GNULIB_ISWXDIGIT",
    "GNULIB_LCHMOD",
    "GNULIB_LCHOWN",
    "GNULIB_LINK",
    "GNULIB_LINKAT",
    "GNULIB_LOCALECONV",
    "GNULIB_LOCALENAME",
    "GNULIB_LOCALTIME",
    "GNULIB_LSEEK",
    "GNULIB_LSTAT",
    "GNULIB_MALLOC_GNU",
    "GNULIB_MALLOC_POSIX",
    "GNULIB_MBRLEN",
    "GNULIB_MBRTOWC",
    "GNULIB_MBSCASECMP",
    "GNULIB_MBSCASESTR",
    "GNULIB_MBSCHR",
    "GNULIB_MBSCSPN",
    "GNULIB_MBSINIT",
    "GNULIB_MBSLEN",
    "GNULIB_MBSNCASECMP",
    "GNULIB_MBSNLEN",
    "GNULIB_MBSNRTOWCS",
    "GNULIB_MBSPBRK",
    "GNULIB_MBSPCASECMP",
    "GNULIB_MBSRCHR",
    "GNULIB_MBSRTOWCS",
    "GNULIB_MBSSEP",
    "GNULIB_MBSSPN",
    "GNULIB_MBSSTR",
    "GNULIB_MBSTOK_R",
    "GNULIB_MBTOWC",
    "GNULIB_MDA_ACCESS",
    "GNULIB_MDA_CHDIR",
    "GNULIB_MDA_CHMOD",
    "GNULIB_MDA_CLOSE",
    "GNULIB_MDA_CREAT",
    "GNULIB_MDA_DUP",
    "GNULIB_MDA_ECVT",
    "GNULIB_MDA_EXECL",
    "GNULIB_MDA_EXECLE",
    "GNULIB_MDA_EXECLP",
    "GNULIB_MDA_EXECV",
    "GNULIB_MDA_EXECVE",
    "GNULIB_MDA_EXECVP",
    "GNULIB_MDA_EXECVPE",
    "GNULIB_MDA_FCLOSEALL",
    "GNULIB_MDA_FCVT",
    "GNULIB_MDA_FDOPEN",
    "GNULIB_MDA_FILENO",
    "GNULIB_MDA_GCVT",
    "GNULIB_MDA_GETCWD",
    "GNULIB_MDA_GETPID",
    "GNULIB_MDA_GETW",
    "GNULIB_MDA_ISATTY",
    "GNULIB_MDA_LSEEK",
    "GNULIB_MDA_MEMCCPY",
    "GNULIB_MDA_MKDIR",
    "GNULIB_MDA_MKTEMP",
    "GNULIB_MDA_PUTENV",
    "GNULIB_MDA_PUTW",
    "GNULIB_MDA_READ",
    "GNULIB_MDA_RMDIR",
    "GNULIB_MDA_STRDUP",
    "GNULIB_MDA_SWAB",
    "GNULIB_MDA_TEMPNAM",
    "GNULIB_MDA_TZSET",
    "GNULIB_MDA_UMASK",
    "GNULIB_MDA_UNLINK",
    "GNULIB_MDA_WCSDUP",
    "GNULIB_MDA_WRITE",
    "GNULIB_MEMCHR",
    "GNULIB_MEMMEM",
    "GNULIB_MEMPCPY",
    "GNULIB_MEMRCHR",
    "GNULIB_MKDIR",
    "GNULIB_MKDIRAT",
    "GNULIB_MKDTEMP",
    "GNULIB_MKFIFO",
    "GNULIB_MKFIFOAT",
    "GNULIB_MKNOD",
    "GNULIB_MKNODAT",
    "GNULIB_MKOSTEMP",
    "GNULIB_MKOSTEMPS",
    "GNULIB_MKSTEMP",
    "GNULIB_MKSTEMPS",
    "GNULIB_MKTIME",
    "GNULIB_NANOSLEEP",
    "GNULIB_NL_LANGINFO",
    "GNULIB_NONBLOCKING",
    "GNULIB_OBSTACK_PRINTF_POSIX",
    "GNULIB_OBSTACK_PRINTF",
    "GNULIB_OPEN",
    "GNULIB_OPENAT",
    "GNULIB_OVERRIDES_STRUCT_STAT",
    "GNULIB_PCLOSE",
    "GNULIB_PERROR",
    "GNULIB_PIPE",
    "GNULIB_PIPE2",
    "GNULIB_POPEN",
    "GNULIB_POSIX_MEMALIGN",
    "GNULIB_POSIX_OPENPT",
    "GNULIB_PREAD",
    "GNULIB_PRINTF_POSIX",
    "GNULIB_PRINTF",
    "GNULIB_PTSNAME_R",
    "GNULIB_PTSNAME",
    "GNULIB_PUTC",
    "GNULIB_PUTCHAR",
    "GNULIB_PUTENV",
    "GNULIB_PUTS",
    "GNULIB_PWRITE",
    "GNULIB_QSORT_R",
    "GNULIB_RANDOM_R",
    "GNULIB_RANDOM",
    "GNULIB_RAWMEMCHR",
    "GNULIB_READ",
    "GNULIB_READLINK",
    "GNULIB_READLINKAT",
    "GNULIB_REALLOC_GNU",
    "GNULIB_REALLOC_POSIX",
    "GNULIB_REALLOCARRAY",
    "GNULIB_REALPATH",
    "GNULIB_REMOVE",
    "GNULIB_RENAME",
    "GNULIB_RENAMEAT",
    "GNULIB_RMDIR",
    "GNULIB_RPMATCH",
    "GNULIB_SCANF",
    "GNULIB_SECURE_GETENV",
    "GNULIB_SETENV",
    "GNULIB_SETHOSTNAME",
    "GNULIB_SETLOCALE_NULL",
    "GNULIB_SETLOCALE",
    "GNULIB_SIGABBREV_NP",
    "GNULIB_SIGDESCR_NP",
    "GNULIB_SLEEP",
    "GNULIB_SNPRINTF",
    "GNULIB_SPRINTF_POSIX",
    "GNULIB_STAT",
    "GNULIB_STDIO_H_NONBLOCKING",
    "GNULIB_STDIO_H_SIGPIPE",
    "GNULIB_STPCPY",
    "GNULIB_STPNCPY",
    "GNULIB_STRCASESTR",
    "GNULIB_STRCHRNUL",
    "GNULIB_STRDUP",
    "GNULIB_STRERROR_R",
    "GNULIB_STRERROR",
    "GNULIB_STRERRORNAME_NP",
    "GNULIB_STRFTIME",
    "GNULIB_STRNCAT",
    "GNULIB_STRNDUP",
    "GNULIB_STRNLEN",
    "GNULIB_STRPBRK",
    "GNULIB_STRPTIME",
    "GNULIB_STRSEP",
    "GNULIB_STRSIGNAL",
    "GNULIB_STRSTR",
    "GNULIB_STRTOD",
    "GNULIB_STRTOK_R",
    "GNULIB_STRTOL",
    "GNULIB_STRTOLD",
    "GNULIB_STRTOLL",
    "GNULIB_STRTOUL",
    "GNULIB_STRTOULL",
    "GNULIB_STRVERSCMP",
    "GNULIB_SYMLINK",
    "GNULIB_SYMLINKAT",
    "GNULIB_SYSTEM_POSIX",
    "GNULIB_TIME_R",
    "GNULIB_TIME_RZ",
    "GNULIB_TIMEGM",
    "GNULIB_TIMESPEC_GET",
    "GNULIB_TIMESPEC_GETRES",
    "GNULIB_TMPFILE",
    "GNULIB_TOWCTRANS",
    "GNULIB_TRUNCATE",
    "GNULIB_TTYNAME_R",
    "GNULIB_TZSET",
    "GNULIB_UNISTD_H_GETOPT",
    "GNULIB_UNLINK",
    "GNULIB_UNLINKAT",
    "GNULIB_UNLOCKPT",
    "GNULIB_UNSETENV",
    "GNULIB_USLEEP",
    "GNULIB_UTIMENSAT",
    "GNULIB_VASPRINTF",
    "GNULIB_VDPRINTF",
    "GNULIB_VFPRINTF_POSIX",
    "GNULIB_VFPRINTF",
    "GNULIB_VFSCANF",
    "GNULIB_VPRINTF_POSIX",
    "GNULIB_VPRINTF",
    "GNULIB_VSCANF",
    "GNULIB_VSNPRINTF",
    "GNULIB_VSPRINTF_POSIX",
    "GNULIB_WCPCPY",
    "GNULIB_WCPNCPY",
    "GNULIB_WCRTOMB",
    "GNULIB_WCSCASECMP",
    "GNULIB_WCSCAT",
    "GNULIB_WCSCHR",
    "GNULIB_WCSCMP",
    "GNULIB_WCSCOLL",
    "GNULIB_WCSCPY",
    "GNULIB_WCSCSPN",
    "GNULIB_WCSDUP",
    "GNULIB_WCSFTIME",
    "GNULIB_WCSLEN",
    "GNULIB_WCSNCASECMP",
    "GNULIB_WCSNCAT",
    "GNULIB_WCSNCMP",
    "GNULIB_WCSNCPY",
    "GNULIB_WCSNLEN",
    "GNULIB_WCSNRTOMBS",
    "GNULIB_WCSPBRK",
    "GNULIB_WCSRCHR",
    "GNULIB_WCSRTOMBS",
    "GNULIB_WCSSPN",
    "GNULIB_WCSSTR",
    "GNULIB_WCSTOK",
    "GNULIB_WCSWIDTH",
    "GNULIB_WCSXFRM",
    "GNULIB_WCTOB",
    "GNULIB_WCTOMB",
    "GNULIB_WCTRANS",
    "GNULIB_WCTYPE",
    "GNULIB_WCWIDTH",
    "GNULIB_WMEMCHR",
    "GNULIB_WMEMCMP",
    "GNULIB_WMEMCPY",
    "GNULIB_WMEMMOVE",
    "GNULIB_WMEMPCPY",
    "GNULIB_WMEMSET",
    "GNULIB_WRITE",
    "GNULIBHEADERS_OVERRIDE_WINT_T",
    "GUARD_PREFIX",
    "HAVE_ALLOCA_H",
    "HAVE_BTOWC",
    "HAVE_C99_STDINT_H",
    "HAVE_CANONICALIZE_FILE_NAME",
    "HAVE_DECL_ECVT",
    "HAVE_DECL_EXECVPE",
    "HAVE_DECL_FCLOSEALL",
    "HAVE_DECL_FCVT",
    "HAVE_DECL_GCVT",
    "HAVE_DECL_GETDELIM",
    "HAVE_DECL_MEMRCHR",
    "HAVE_DECL_UNSETENV",
    "HAVE_DECL_WCSDUP",
    "HAVE_DECL_WCTOB",
    "HAVE_DUPLOCALE",
    "HAVE_EXECVPE",
    "HAVE_FCNTL",
    "HAVE_FEATURES_H",
    "HAVE_FREELOCALE",
    "HAVE_GETDTABLESIZE",
    "HAVE_GETRANDOM",
    "HAVE_ISWBLANK",
    "HAVE_ISWCNTRL",
    "HAVE_LANGINFO_ALTMON",
    "HAVE_LANGINFO_CODESET",
    "HAVE_LANGINFO_ERA",
    "HAVE_LANGINFO_H",
    "HAVE_LANGINFO_T_FMT_AMPM",
    "HAVE_LANGINFO_YESEXPR",
    "HAVE_LSTAT",
    "HAVE_MBRLEN",
    "HAVE_MBRTOWC",
    "HAVE_MBSINIT",
    "HAVE_MBTOWC",
    "HAVE_MEMPCPY",
    "HAVE_MKOSTEMP",
    "HAVE_NEWLOCALE",
    "HAVE_NL_LANGINFO",
    "HAVE_RAWMEMCHR",
    "HAVE_READLINK",
    "HAVE_REALLOCARRAY",
    "HAVE_REALPATH",
    "HAVE_STDINT_H",
    "HAVE_STRVERSCMP",
    "HAVE_SYS_LOADAVG_H",
    "HAVE_SYS_RANDOM_H",
    "HAVE_TIMEZONE_T",
    "HAVE_UNISTD_H",
    "HAVE_WCHAR_H",
    "HAVE_WCRTOMB",
    "HAVE_WCTRANS_T",
    "HAVE_WCTYPE_H",
    "HAVE_WCTYPE_T",
    "HAVE_WINT_T",
    "HAVE_XLOCALE_H",
    "INCLUDE_NEXT",
    "LOCALENAME_ENHANCE_LOCALE_FUNCS",
    "NEXT_ASSERT_H",
    "NEXT_CTYPE_H",
    "NEXT_FCNTL_H",
    "NEXT_LANGINFO_H",
    "NEXT_LIMITS_H",
    "NEXT_LOCALE_H",
    "NEXT_STDINT_H",
    "NEXT_STDIO_H",
    "NEXT_STDLIB_H",
    "NEXT_STRING_H",
    "NEXT_SYS_RANDOM_H",
    "NEXT_SYS_STAT_H",
    "NEXT_SYS_TYPES_H",
    "NEXT_TIME_H",
    "NEXT_UNISTD_H",
    "NEXT_WCHAR_H",
    "NEXT_WCTYPE_H",
    "PRAGMA_COLUMNS",
    "PRAGMA_SYSTEM_HEADER",
    "REPLACE_BTOWC",
    "REPLACE_CALLOC_FOR_CALLOC_GNU",
    "REPLACE_CALLOC_FOR_CALLOC_POSIX",
    "REPLACE_CANONICALIZE_FILE_NAME",
    "REPLACE_CLOSE",
    "REPLACE_DUP2",
    "REPLACE_FCNTL",
    "REPLACE_FPRINTF",
    "REPLACE_FREE",
    "REPLACE_FSTAT",
    "REPLACE_GETDELIM",
    "REPLACE_GETDTABLESIZE",
    "REPLACE_GETRANDOM",
    "REPLACE_GMTIME",
    "REPLACE_ISWBLANK",
    "REPLACE_ISWCNTRL",
    "REPLACE_LOCALECONV",
    "REPLACE_LOCALTIME",
    "REPLACE_LSTAT",
    "REPLACE_MALLOC_FOR_MALLOC_GNU",
    "REPLACE_MALLOC_FOR_MALLOC_POSIX",
    "REPLACE_MBRLEN",
    "REPLACE_MBRTOWC",
    "REPLACE_MBSINIT",
    "REPLACE_MBSTATE_T",
    "REPLACE_MBTOWC",
    "REPLACE_MEMCHR",
    "REPLACE_MKDIR",
    "REPLACE_NL_LANGINFO",
    "REPLACE_OPEN",
    "REPLACE_PRINTF",
    "REPLACE_READLINK",
    "REPLACE_REALLOC_FOR_REALLOC_GNU",
    "REPLACE_REALLOC_FOR_REALLOC_POSIX",
    "REPLACE_REALLOCARRAY",
    "REPLACE_REALPATH",
    "REPLACE_RENAME",
    "REPLACE_RMDIR",
    "REPLACE_STAT",
    "REPLACE_STDIO_READ_FUNCS",
    "REPLACE_STDIO_WRITE_FUNCS",
    "REPLACE_STRERROR",
    "REPLACE_STRUCT_LCONV",
    "REPLACE_TOWLOWER",
    "REPLACE_VFPRINTF",
    "REPLACE_VPRINTF",
    "REPLACE_WCRTOMB",
    "REPLACE_WCTOB",
    "REPLACE_WCTOMB",
    "TIME_H_DEFINES_STRUCT_TIMESPEC",
    "TIME_H_DEFINES_TIME_UTC",
    "UNISTD_H_HAVE_SYS_RANDOM_H",
    "UNISTD_H_HAVE_WINSOCK2_H_AND_USE_SOCKETS",
    "UNISTD_H_HAVE_WINSOCK2_H",
    "WINDOWS_64_BIT_OFF_T",
    "WINDOWS_64_BIT_ST_SIZE",
    "WINDOWS_STAT_INODES",
}


def load_stat_variables(config_status: Path) -> dict[str, str]:
    results = {}
    pattern = re.compile(r'S\["([\w_]+)"\]="(.*)"')
    for line in config_status.read_text(encoding="utf-8").splitlines():
        matches = pattern.match(line)
        if matches:
            results[matches.group(1)] = matches.group(2)

    return results


_TEMPLATE = """\
\"\"\"Platform specific configuration values\"\"\"

CONFIGURATION = {}
"""


def main() -> None:
    config_h = Path("config.h")
    config_status = Path("config.status")

    variables = load_stat_variables(config_status)

    # insert known missing variables
    variables["GUARD_PREFIX"] = "GL"

    filtered = {}
    for value in KNOWN_VALUES:
        if value in variables:
            filtered[f"@{value}@"] = variables[value]
        elif f"GL_{value}" in variables:
            filtered[f"@{value}@"] = variables[f"GL_{value}"]
        else:
            print("MISSING:", value)

    plat_name = "linux"
    if platform.system() == "Windows":
        plat_name = "windows"
    if platform.system() == "Darwin":
        plat_name = "macos"

    shutil.copy2(config_h, config_h.parent / config_h.name.replace(".h", f"_{plat_name}.h"))
    Path(f"config_{plat_name}.bzl").write_text(
        _TEMPLATE.format(json.dumps(filtered, indent=4, sort_keys=True)),
        encoding="utf-8",
    )
    print("Done")


if __name__ == "__main__":
    main()
```
