// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by consensys/gnark-crypto DO NOT EDIT

package fft

import (
	"math/big"
	"strconv"
	"testing"

	"github.com/consensys/gnark-crypto/field/koalabear"
	fext "github.com/consensys/gnark-crypto/field/koalabear/extensions"

	"github.com/leanovate/gopter"
	"github.com/leanovate/gopter/gen"
	"github.com/leanovate/gopter/prop"

	"encoding/binary"
	"fmt"
	"github.com/stretchr/testify/require"
	"math/rand/v2"
)

func TestFFTExt(t *testing.T) {
	parameters := gopter.DefaultTestParameters()
	parameters.MinSuccessfulTests = 6
	properties := gopter.NewProperties(parameters)

	for maxSize := 2; maxSize <= 1<<10; maxSize <<= 1 {

		domainWithPrecompute := NewDomain(uint64(maxSize))
		domainWithoutPrecompute := NewDomain(uint64(maxSize), WithoutPrecompute())

		for domainName, domain := range map[string]*Domain{
			"with precompute":    domainWithPrecompute,
			"without precompute": domainWithoutPrecompute,
		} {
			domainName := domainName
			domain := domain
			t.Logf("domain: %s", domainName)
			properties.Property("DIF FFT should be consistent with dual basis", prop.ForAll(

				// checks that a random evaluation of a dual function eval(gen**ithpower) is consistent with the FFT result
				func(ithpower int) bool {

					pol := make([]fext.E4, maxSize)
					backupPol := make([]fext.E4, maxSize)

					for i := 0; i < maxSize; i++ {
						pol[i].MustSetRandom()
					}
					copy(backupPol, pol)

					domain.FFTExt(pol, DIF)
					BitReverse(pol)

					sample := domain.Generator
					sample.Exp(sample, big.NewInt(int64(ithpower)))

					eval := evaluatePolynomialExt(backupPol, sample)

					return eval.Equal(&pol[ithpower])

				},
				gen.IntRange(0, maxSize-1),
			))

			properties.Property("DIF FFT on cosets should be consistent with dual basis", prop.ForAll(

				// checks that a random evaluation of a dual function eval(gen**ithpower) is consistent with the FFT result
				func(ithpower int) bool {

					pol := make([]fext.E4, maxSize)
					backupPol := make([]fext.E4, maxSize)

					for i := 0; i < maxSize; i++ {
						pol[i].MustSetRandom()
					}
					copy(backupPol, pol)

					domain.FFTExt(pol, DIF, OnCoset())
					BitReverse(pol)

					sample := domain.Generator
					sample.Exp(sample, big.NewInt(int64(ithpower))).
						Mul(&sample, &domain.FrMultiplicativeGen)

					eval := evaluatePolynomialExt(backupPol, sample)

					return eval.Equal(&pol[ithpower])

				},
				gen.IntRange(0, maxSize-1),
			))

			properties.Property("DIT FFT should be consistent with dual basis", prop.ForAll(

				// checks that a random evaluation of a dual function eval(gen**ithpower) is consistent with the FFT result
				func(ithpower int) bool {

					pol := make([]fext.E4, maxSize)
					backupPol := make([]fext.E4, maxSize)

					for i := 0; i < maxSize; i++ {
						pol[i].MustSetRandom()
					}
					copy(backupPol, pol)

					BitReverse(pol)
					domain.FFTExt(pol, DIT)

					sample := domain.Generator
					sample.Exp(sample, big.NewInt(int64(ithpower)))

					eval := evaluatePolynomialExt(backupPol, sample)

					return eval.Equal(&pol[ithpower])

				},
				gen.IntRange(0, maxSize-1),
			))

			properties.Property("bitReverse(DIF FFT(DIT FFT (bitReverse))))==id", prop.ForAll(

				func() bool {

					pol := make([]fext.E4, maxSize)
					backupPol := make([]fext.E4, maxSize)

					for i := 0; i < maxSize; i++ {
						pol[i].MustSetRandom()
					}
					copy(backupPol, pol)

					BitReverse(pol)
					domain.FFTExt(pol, DIT)
					domain.FFTInverseExt(pol, DIF)
					BitReverse(pol)

					check := true
					for i := 0; i < len(pol); i++ {
						check = check && pol[i].Equal(&backupPol[i])
					}
					return check
				},
			))

			for nbCosets := 2; nbCosets < 5; nbCosets++ {
				properties.Property(fmt.Sprintf("bitReverse(DIF FFT(DIT FFT (bitReverse))))==id on %d cosets", nbCosets), prop.ForAll(

					func() bool {

						pol := make([]fext.E4, maxSize)
						backupPol := make([]fext.E4, maxSize)

						for i := 0; i < maxSize; i++ {
							pol[i].MustSetRandom()
						}
						copy(backupPol, pol)

						check := true

						for i := 1; i <= nbCosets; i++ {

							BitReverse(pol)
							domain.FFTExt(pol, DIT, OnCoset())
							domain.FFTInverseExt(pol, DIF, OnCoset())
							BitReverse(pol)

							for i := 0; i < len(pol); i++ {
								check = check && pol[i].Equal(&backupPol[i])
							}
						}

						return check
					},
				))
			}

			properties.Property("DIT FFT(DIF FFT)==id", prop.ForAll(

				func() bool {

					pol := make([]fext.E4, maxSize)
					backupPol := make([]fext.E4, maxSize)

					for i := 0; i < maxSize; i++ {
						pol[i].MustSetRandom()
					}
					copy(backupPol, pol)

					domain.FFTInverseExt(pol, DIF)
					domain.FFTExt(pol, DIT)

					check := true
					for i := 0; i < len(pol); i++ {
						check = check && (pol[i] == backupPol[i])
					}
					return check
				},
			))

			properties.Property("DIT FFT(DIF FFT)==id on cosets", prop.ForAll(

				func() bool {

					pol := make([]fext.E4, maxSize)
					backupPol := make([]fext.E4, maxSize)

					for i := 0; i < maxSize; i++ {
						pol[i].MustSetRandom()
					}
					copy(backupPol, pol)

					domain.FFTInverseExt(pol, DIF, OnCoset())
					domain.FFTExt(pol, DIT, OnCoset())

					for i := 0; i < len(pol); i++ {
						if !(pol[i].Equal(&backupPol[i])) {
							return false
						}
					}

					// compute with nbTasks == 1
					domain.FFTInverseExt(pol, DIF, OnCoset(), WithNbTasks(1))
					domain.FFTExt(pol, DIT, OnCoset(), WithNbTasks(1))

					for i := 0; i < len(pol); i++ {
						if !(pol[i].Equal(&backupPol[i])) {
							return false
						}
					}

					return true
				},
			))
		}
		properties.TestingRun(t, gopter.ConsoleReporter(false))
	}

}

func randElementExt(rng *rand.Rand) fext.E4 {
	var v fext.E4
	v.B0.A0 = koalabear.Element{rng.Uint32N(2130706433)}
	v.B0.A1 = koalabear.Element{rng.Uint32N(2130706433)}
	v.B1.A0 = koalabear.Element{rng.Uint32N(2130706433)}
	v.B1.A1 = koalabear.Element{rng.Uint32N(2130706433)}
	return v
}

func FuzzFFTExt(f *testing.F) {
	f.Fuzz(func(t *testing.T, domainSize uint16, rngSeed int64) {
		if domainSize > (1 << 13) {
			t.Skip("domain size too large")
		}
		if domainSize < 2 {
			t.Skip("domain size too small")
		}

		domain := NewDomain(uint64(domainSize))

		var seed [32]byte
		binary.PutVarint(seed[:], rngSeed)
		// #nosec G404 -- fuzz does not require a cryptographic PRNG
		rng := rand.New(rand.NewChaCha8(seed))

		cardinality := domain.Cardinality

		// we just check that FFT-1(FFT(pol)) == pol
		a, b := make([]fext.E4, cardinality), make([]fext.E4, cardinality)
		for i := 0; i < int(cardinality); i++ {
			a[i] = randElementExt(rng)
		}
		copy(b, a)

		domain.FFTInverseExt(a, DIF)
		domain.FFTExt(a, DIT)

		assert := require.New(t)
		for i := 0; i < int(cardinality); i++ {
			assert.True(a[i].Equal(&b[i]), "FFT-1(FFT(pol)) != pol at index %d", i)
		}
	})
}

// --------------------------------------------------------------------
// benches

func BenchmarkFFTExt(b *testing.B) {

	const maxSize = 1 << 20

	pol := make([]fext.E4, maxSize)
	pol[0].MustSetRandom()
	for i := 1; i < maxSize; i++ {
		pol[i] = pol[i-1]
	}

	for i := 8; i < 20; i++ {
		sizeDomain := 1 << i
		b.Run("fft 2**"+strconv.Itoa(i)+"bits", func(b *testing.B) {
			domain := NewDomain(uint64(sizeDomain))
			b.ResetTimer()
			for j := 0; j < b.N; j++ {
				domain.FFTExt(pol[:sizeDomain], DIT)
			}
		})
		b.Run("fft 2**"+strconv.Itoa(i)+"bits (coset)", func(b *testing.B) {
			domain := NewDomain(uint64(sizeDomain))
			b.ResetTimer()
			for j := 0; j < b.N; j++ {
				domain.FFTExt(pol[:sizeDomain], DIT, OnCoset())
			}
		})
	}

}

func BenchmarkFFTDITCosetReferenceExt(b *testing.B) {
	const maxSize = 1 << 20

	pol := make([]fext.E4, maxSize)
	pol[0].MustSetRandom()
	for i := 1; i < maxSize; i++ {
		pol[i] = pol[i-1]
	}

	domain := NewDomain(maxSize)

	b.ResetTimer()
	for j := 0; j < b.N; j++ {
		domain.FFTExt(pol, DIT, OnCoset())
	}
}

func BenchmarkFFTDITReferenceSmallExt(b *testing.B) {
	const maxSize = 1 << 9

	pol := make([]fext.E4, maxSize)
	pol[0].MustSetRandom()
	for i := 1; i < maxSize; i++ {
		pol[i] = pol[i-1]
	}

	domain := NewDomain(maxSize)

	b.ResetTimer()
	for j := 0; j < 1; j++ {
		domain.FFTExt(pol, DIT)
	}
}

func BenchmarkFFTDIFReferenceExt(b *testing.B) {
	const maxSize = 1 << 20

	pol := make([]fext.E4, maxSize)
	pol[0].MustSetRandom()
	for i := 1; i < maxSize; i++ {
		pol[i] = pol[i-1]
	}

	domain := NewDomain(maxSize)

	b.ResetTimer()
	for j := 0; j < b.N; j++ {
		domain.FFTExt(pol, DIF)
	}
}
func BenchmarkFFTDIFReferenceSmallExt(b *testing.B) {
	const maxSize = 1 << 9

	pol := make([]fext.E4, maxSize)
	pol[0].MustSetRandom()
	for i := 1; i < maxSize; i++ {
		pol[i] = pol[i-1]
	}

	domain := NewDomain(maxSize)

	b.ResetTimer()
	for j := 0; j < b.N; j++ {
		domain.FFTExt(pol, DIF)
	}
}

func evaluatePolynomialExt(pol []fext.E4, val koalabear.Element) fext.E4 {
	var res, tmp fext.E4
	var acc koalabear.Element
	res.Set(&pol[0])
	acc.Set(&val)
	for i := 1; i < len(pol); i++ {
		tmp.MulByElement(&pol[i], &acc)
		res.Add(&res, &tmp)
		acc.Mul(&acc, &val)
	}
	return res
}
