/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef MASSADD_DIALOG_H
#define MASSADD_DIALOG_H

#include "gui/dialog.h"
#include "gui/widgets/list.h"
#include "common/fs.h"
#include "common/hashmap.h"
#include "common/stack.h"
#include "common/str.h"

namespace GUI {

class StaticTextWidget;
class MassAddListWidget;

class MassAddDialog : public Dialog {
public:
	MassAddDialog(const Common::FSNode &startDir);

	//void open();
	void handleCommand(CommandSender *sender, uint32 cmd, uint32 data) override;
	void handleTickle() override;

	Common::String getFirstAddedTarget() const {
		if (!_games.empty())
			return _games.front().gameId;
		return Common::String();
	}

private:
	Common::Stack<Common::FSNode>  _scanStack;
	DetectedGames _games;

	void updateGameList();

	/**
	 * Map each path occurring in the config file to the target(s) using that path.
	 * Used to detect whether a potential new target is already present in the
	 * config manager.
	 */
	Common::HashMap<Common::Path, Common::StringArray,
		Common::Path::IgnoreCase_Hash, Common::Path::IgnoreCase_EqualTo> _pathToTargets;

	int _dirsScanned;
	int _oldGamesCount;
	int _dirTotal;

	Widget *_okButton;
	StaticTextWidget *_dirProgressText;
	StaticTextWidget *_gameProgressText;

	MassAddListWidget *_list;
};

class MassAddListWidget : public ListWidget {
public:
	MassAddListWidget(Dialog *boss, const Common::String &name)
		: ListWidget(boss, name) { }

	void appendToSelectedList(bool selected) { _listSelected.push_back(selected); }
	void clearSelectedList() { _listSelected.clear(); }

protected:
	ThemeEngine::WidgetStateInfo getItemState(int item) const override {
		// Display selected/unselected games in mass detection as enabled/disabled items.
		if (item < (signed int)_listSelected.size() && _listSelected[item]) {
			return ThemeEngine::kStateEnabled;
		} else {
			return ThemeEngine::kStateDisabled;
		}
	}

	Common::Array<bool>	_listSelected;
};

} // End of namespace GUI

#endif
