//
// Syd: rock-solid application kernel
// src/kernel/sigreturn.rs: {,_rt}sigreturn(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use data_encoding::HEXLOWER;
use memchr::arch::all::is_equal;
use nix::errno::Errno;

use crate::{
    cache::SigreturnResult,
    config::MMAP_MIN_ADDR,
    elf::{disasm, scmp_syscall_instruction, scmp_sysret_instruction},
    error,
    hook::RemoteProcess,
    proc::proc_maps,
    ptrace::ptrace_syscall_info,
    sandbox::Action,
    scmp_arch, SydArch,
};

// Note sigreturn is a ptrace(2) hook, not a seccomp hook!
#[allow(clippy::cognitive_complexity)]
pub(crate) fn sysexit_sigreturn(
    process: RemoteProcess,
    info: ptrace_syscall_info,
    result: SigreturnResult,
) -> Result<(), Errno> {
    let is_realtime = result.is_realtime;
    let args = result.args;
    let ip_entry = result.ip;
    let sp_entry = result.sp;
    let ip_entry_mem = result.ip_mem;
    let sp_entry_mem = result.sp_mem;

    let mut error: Option<&'static str> = None;

    // SAFETY: Check if stack pointer is invalid.
    if info.stack_pointer < *MMAP_MIN_ADDR || info.instruction_pointer < *MMAP_MIN_ADDR {
        error = Some("stack smashing detected!");
    }

    let mut ip_mem = [0u8; 64];
    let mut ip_read = false;

    // SAFETY: Check for a syscall instruction at memory pointed by ip.
    if error.is_some() {
        // SAFETY: We validate the PidFd after memory read.
        match unsafe { process.read_mem(&mut ip_mem, info.instruction_pointer) } {
            Ok(_) if !process.is_alive() => return Err(Errno::ESRCH),
            Ok(_) => ip_read = true,
            Err(Errno::ESRCH) => return Err(Errno::ESRCH),
            Err(_) => {
                // SAFETY: Process is alive, but
                // we cannot read memory: Terminate!
                let _ = process.pidfd_kill(libc::SIGKILL);
                return Err(Errno::ESRCH);
            }
        }
    }

    let sys_instr = scmp_syscall_instruction(info.arch);
    let sys_instr_len = sys_instr.len();
    if sys_instr_len == 0 {
        // SAFETY: Unsupported architecture, continue process.
        return Ok(());
    }

    if error.is_none() && is_equal(&sys_instr[..sys_instr_len], &ip_mem[..sys_instr_len]) {
        error = Some("SROP detected!");
    }

    let error = if let Some(error) = error {
        error
    } else {
        // SAFETY: No SROP detected, continue process.
        return Ok(());
    };

    let mut sp_mem = [0u8; 64];
    let mut sp_read = false;

    if info.stack_pointer >= *MMAP_MIN_ADDR {
        #[allow(clippy::arithmetic_side_effects)]
        // SAFETY: No validation, data is used for logging only.
        match unsafe { process.read_mem(&mut sp_mem, (info.stack_pointer & !0xF) - 16) } {
            Ok(_) => sp_read = true,
            Err(Errno::ESRCH) => {}
            Err(_) => {
                // SAFETY: Process is alive, but
                // we cannot read memory: Terminate!
                let _ = process.pidfd_kill(libc::SIGKILL);
                return Err(Errno::ESRCH);
            }
        }
    }

    // Read memory maps for logging.
    let memmap = proc_maps(process.pid).ok();

    // SAFETY: SROP detected, terminate process!
    let _ = process.pidfd_kill(libc::SIGKILL);

    // SAFETY: We have checked for supported arch before this point.
    #[allow(clippy::disallowed_methods)]
    let arch = scmp_arch(info.arch).unwrap();

    let ip_asm = if ip_read {
        disasm(&ip_mem, arch, info.instruction_pointer, true, false)
            .map(|instructions| {
                instructions
                    .into_iter()
                    .map(|instruction| instruction.op)
                    .collect::<Vec<_>>()
            })
            .ok()
    } else {
        None
    };

    let ip_entry_asm = if let Some(ip_entry_mem) = ip_entry_mem {
        disasm(&ip_entry_mem, arch, ip_entry, true, false)
            .map(|instructions| {
                instructions
                    .into_iter()
                    .map(|instruction| instruction.op)
                    .collect::<Vec<_>>()
            })
            .ok()
    } else {
        None
    };

    let ip_mem = if ip_read {
        Some(HEXLOWER.encode(&ip_mem))
    } else {
        None
    };

    let sp_mem = if sp_read {
        Some(HEXLOWER.encode(&sp_mem))
    } else {
        None
    };

    let ip_entry_mem = ip_entry_mem.map(|ip_entry_mem| HEXLOWER.encode(&ip_entry_mem));
    let sp_entry_mem = sp_entry_mem.map(|sp_entry_mem| HEXLOWER.encode(&sp_entry_mem));

    #[allow(clippy::disallowed_methods)]
    let arch = SydArch(scmp_arch(info.arch).unwrap());
    error!("ctx": "sigreturn", "op": "check_SROP",
        "err": error,
        "act": Action::Kill,
        "pid": process.pid.as_raw(),
        "sys": if is_realtime { "rt_sigreturn" } else { "sigreturn" },
        "args": args,
        "arch": arch,
        "ret": HEXLOWER.encode(scmp_sysret_instruction(info.arch)),
        "ip": info.instruction_pointer,
        "sp": info.stack_pointer,
        "ip_entry": ip_entry,
        "sp_entry": sp_entry,
        "ip_asm": ip_asm,
        "ip_entry_asm": ip_entry_asm,
        "ip_mem": ip_mem,
        "sp_mem": sp_mem,
        "ip_entry_mem": ip_entry_mem,
        "sp_entry_mem": sp_entry_mem,
        "memmap": memmap);

    Err(Errno::ESRCH)
}
