// libsemigroups - C++ library for semigroups and monoids
// Copyright (C) 2022 Reinis Cirpons
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

// These tests are derived from:

#include <cstddef>  // for size_t
#include <cstdint>  // for uint64_t

#include "catch.hpp"                 // for LIBSEMIGROUPS_TEST_CASE
#include "libsemigroups/report.hpp"  // for ReportGuard
#include "libsemigroups/schreier-sims-helper.hpp"  // For schreier_sim_helper::intersection
#include "libsemigroups/schreier-sims.hpp"  // for SchreierSims, SchreierSims<>::ele...
#include "libsemigroups/transf.hpp"         // for Perm
#include "test-main.hpp"                    // for LIBSEMIGROUPS_TEST_CASE

namespace libsemigroups {
  struct LibsemigroupsException;

  constexpr bool REPORT = false;

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "001",
                          "trivial perm. group intersection (degree 1)",
                          "[quick][schreier-sims-helper]") {
    auto            rg = ReportGuard(REPORT);
    SchreierSims<1> S, T, U;
    using Perm = decltype(S)::element_type;
    S.add_generator(Perm({0}));
    T.add_generator(Perm({0}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 1);
    REQUIRE(U.contains(Perm({0})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "002",
                          "trivial perm. group intersection (degree 2)",
                          "[quick][schreier-sims]") {
    auto            rg = ReportGuard(REPORT);
    SchreierSims<2> S, T, U;
    using Perm = SchreierSims<2>::element_type;
    S.add_generator(Perm({0, 1}));
    T.add_generator(Perm({1, 0}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 1);
    REQUIRE(U.sift(Perm({1, 0})) == Perm({1, 0}));
    REQUIRE(!U.contains(Perm({1, 0})));
    REQUIRE(U.contains(Perm({0, 1})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "003",
                          "cyclic group intersection (degree 13)",
                          "[quick][schreier-sims]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<13> S, T, U;
    using Perm = SchreierSims<13>::element_type;
    // Adapted from:
    // https://math.stackexchange.com/q/4093199/152276
    // (0, 1, 2, 3, 4)(5, 9)(6, 10)(7, 11)(8, 12)
    S.add_generator(Perm({1, 2, 3, 4, 0, 9, 10, 11, 12, 5, 6, 7, 8}));
    // (1, 4)(2, 3)(5, 6, 7, 8, 9, 10, 11, 12)
    T.add_generator(Perm({0, 4, 3, 2, 1, 6, 7, 8, 9, 10, 11, 12, 5}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 2);
    REQUIRE(!U.contains(Perm({1, 0, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12})));
    REQUIRE(U.contains(Perm({0, 1, 2, 3, 4, 9, 10, 11, 12, 5, 6, 7, 8})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "004",
                          "D10 and Z5 intersection",
                          "[quick][schreier-sims]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<12> S, T, U;
    using Perm = SchreierSims<12>::element_type;
    // (0, 2, 4, 6, 8)(1, 3, 5, 7, 9)(10, 11)
    S.add_generator(Perm({2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 11, 10}));
    // (0, 2, 4, 6, 8)(1, 3, 5, 7, 9)
    // (0, 1)(2, 9)(3, 8)(4, 7)(5, 6)
    T.add_generator(Perm({2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 10, 11}));
    T.add_generator(Perm({1, 0, 9, 8, 7, 6, 5, 4, 3, 2, 10, 11}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 5);
    // (0, 6, 2, 8, 4)(1, 7, 3, 9, 5)
    REQUIRE(U.contains(Perm({6, 7, 8, 9, 0, 1, 2, 3, 4, 5, 10, 11})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "005",
                          "D8 and Q8 intersection",
                          "[quick][schreier-sims]") {
    auto            rg = ReportGuard(REPORT);
    SchreierSims<8> S, T, U;
    using Perm = SchreierSims<8>::element_type;
    S.add_generator(Perm({1, 3, 7, 5, 2, 0, 4, 6}));
    S.add_generator(Perm({2, 4, 3, 6, 5, 7, 0, 1}));
    S.add_generator(Perm({3, 5, 6, 0, 7, 1, 2, 4}));
    T.add_generator(Perm({1, 0, 7, 5, 6, 3, 4, 2}));
    T.add_generator(Perm({2, 4, 3, 6, 5, 7, 0, 1}));
    T.add_generator(Perm({3, 5, 6, 0, 7, 1, 2, 4}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 4);
    REQUIRE(U.contains(Perm({2, 4, 3, 6, 5, 7, 0, 1})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "006",
                          "primitive on 8 points intersection",
                          "[quick][schreier-sims]") {
    auto            rg = ReportGuard(REPORT);
    SchreierSims<8> S, T, U;
    using Perm = SchreierSims<8>::element_type;
    S.add_generator(Perm({0, 2, 3, 4, 1, 5, 6, 7}));
    S.add_generator(Perm({1, 2, 4, 0, 3, 5, 6, 7}));
    T.add_generator(Perm({1, 2, 3, 4, 5, 6, 0, 7}));
    T.add_generator(Perm({0, 1, 2, 3, 4, 6, 7, 5}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 10);
    REQUIRE(U.contains(Perm({0, 3, 4, 1, 2, 5, 6, 7})));
    REQUIRE(U.contains(Perm({1, 2, 4, 0, 3, 5, 6, 7})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "007",
                          "primitive on 8 points intersection (swap order)",
                          "[quick][schreier-sims]") {
    auto            rg = ReportGuard(REPORT);
    SchreierSims<8> S, T, U;
    using Perm = SchreierSims<8>::element_type;
    S.add_generator(Perm({0, 2, 3, 4, 1, 5, 6, 7}));
    S.add_generator(Perm({1, 2, 4, 0, 3, 5, 6, 7}));
    T.add_generator(Perm({1, 2, 3, 4, 5, 6, 0, 7}));
    T.add_generator(Perm({0, 1, 2, 3, 4, 6, 7, 5}));
    schreier_sims_helper::intersection(U, T, S);
    REQUIRE(U.size() == 10);
    REQUIRE(U.contains(Perm({0, 3, 4, 1, 2, 5, 6, 7})));
    REQUIRE(U.contains(Perm({1, 2, 4, 0, 3, 5, 6, 7})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "008",
                          "A13 and PGL(2, 11) intersection",
                          "[quick][schreier-sims]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<13> S, T, U;
    using Perm = SchreierSims<13>::element_type;
    S.add_generator(Perm({1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 0}));
    S.add_generator(Perm({0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 11, 12, 10}));
    T.add_generator(Perm({1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 0, 11, 12}));
    T.add_generator(Perm({0, 2, 4, 6, 8, 10, 1, 3, 5, 7, 9, 11, 12}));
    T.add_generator(Perm({11, 10, 5, 7, 8, 2, 9, 3, 4, 6, 1, 0, 12}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 660);
    REQUIRE(U.contains(Perm({1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 0, 11, 12})));
    REQUIRE(U.contains(Perm({0, 4, 8, 1, 5, 9, 2, 6, 10, 3, 7, 11, 12})));
    REQUIRE(U.contains(Perm({11, 10, 5, 7, 8, 2, 9, 3, 4, 6, 1, 0, 12})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "009",
                          "A13 and PGL(2, 11) intersection (swap order)",
                          "[quick][schreier-sims]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<13> S, T, U;
    using Perm = SchreierSims<13>::element_type;
    S.add_generator(Perm({1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 0}));
    S.add_generator(Perm({0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 11, 12, 10}));
    T.add_generator(Perm({1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 0, 11, 12}));
    T.add_generator(Perm({0, 2, 4, 6, 8, 10, 1, 3, 5, 7, 9, 11, 12}));
    T.add_generator(Perm({11, 10, 5, 7, 8, 2, 9, 3, 4, 6, 1, 0, 12}));
    schreier_sims_helper::intersection(U, T, S);
    REQUIRE(U.size() == 660);
    REQUIRE(U.contains(Perm({1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 0, 11, 12})));
    REQUIRE(U.contains(Perm({0, 4, 8, 1, 5, 9, 2, 6, 10, 3, 7, 11, 12})));
    REQUIRE(U.contains(Perm({11, 10, 5, 7, 8, 2, 9, 3, 4, 6, 1, 0, 12})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "010",
                          "S17 and A39 intersection",
                          "[standard][schreier-sims][no-valgrind]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<50> S, T, U;
    using Perm = SchreierSims<50>::element_type;
    S.add_generator(Perm({1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13,
                          14, 15, 16, 0,  17, 18, 19, 20, 21, 22, 23, 24, 25,
                          26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38,
                          39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49}));
    S.add_generator(Perm({1,  0,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12,
                          13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
                          26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38,
                          39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49}));
    T.add_generator(Perm({1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13,
                          14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26,
                          27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 0,
                          39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49}));
    T.add_generator(Perm({0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12,
                          13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
                          26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 37, 38, 36,
                          39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 177843714048000);
    REQUIRE(U.contains(Perm({1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13,
                             14, 15, 16, 0,  17, 18, 19, 20, 21, 22, 23, 24, 25,
                             26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38,
                             39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49})));
    REQUIRE(U.contains(Perm({0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12,
                             13, 15, 16, 14, 17, 18, 19, 20, 21, 22, 23, 24, 25,
                             26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38,
                             39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "011",
                          "A50 and PGL(2, 49) intersection",
                          "[extreme][schreier-sims][no-valgrind]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<50> S, T, U;
    using Perm = SchreierSims<50>::element_type;
    S.add_generator(Perm({1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13,
                          14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26,
                          27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39,
                          40, 41, 42, 43, 44, 45, 46, 47, 48, 0,  49}));
    S.add_generator(Perm({0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12,
                          13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
                          26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38,
                          39, 40, 41, 42, 43, 44, 45, 46, 48, 49, 47}));
    T.add_generator(Perm({1,  2,  3,  4,  5,  6,  0,  8,  9,  10, 11, 12, 13,
                          7,  15, 16, 17, 18, 19, 20, 14, 22, 23, 24, 25, 26,
                          27, 21, 29, 30, 31, 32, 33, 34, 28, 36, 37, 38, 39,
                          40, 41, 35, 43, 44, 45, 46, 47, 48, 42, 49}));
    T.add_generator(Perm({0,  2,  4,  6,  1,  3,  5,  14, 16, 18, 20, 15, 17,
                          19, 28, 30, 32, 34, 29, 31, 33, 42, 44, 46, 48, 43,
                          45, 47, 7,  9,  11, 13, 8,  10, 12, 21, 23, 25, 27,
                          22, 24, 26, 35, 37, 39, 41, 36, 38, 40, 49}));
    T.add_generator(Perm({0,  17, 34, 44, 12, 22, 39, 26, 36, 4,  14, 31, 48,
                          9,  45, 13, 23, 40, 1,  18, 28, 15, 32, 42, 10, 27,
                          37, 5,  41, 2,  19, 29, 46, 7,  24, 11, 21, 38, 6,
                          16, 33, 43, 30, 47, 8,  25, 35, 3,  20, 49}));
    T.add_generator(Perm({49, 6,  3,  2,  5,  4,  1,  7,  31, 22, 41, 36, 27,
                          32, 28, 46, 19, 38, 39, 16, 45, 35, 9,  30, 43, 48,
                          33, 12, 14, 44, 23, 8,  13, 26, 47, 21, 11, 40, 17,
                          18, 37, 10, 42, 24, 29, 20, 15, 34, 25, 0}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 58800);
    REQUIRE(U.contains(Perm({1,  2,  3,  4,  5,  6,  0,  8,  9,  10, 11, 12, 13,
                             7,  15, 16, 17, 18, 19, 20, 14, 22, 23, 24, 25, 26,
                             27, 21, 29, 30, 31, 32, 33, 34, 28, 36, 37, 38, 39,
                             40, 41, 35, 43, 44, 45, 46, 47, 48, 42, 49})));
    REQUIRE(U.contains(Perm({0,  2,  4,  6,  1,  3,  5,  14, 16, 18, 20, 15, 17,
                             19, 28, 30, 32, 34, 29, 31, 33, 42, 44, 46, 48, 43,
                             45, 47, 7,  9,  11, 13, 8,  10, 12, 21, 23, 25, 27,
                             22, 24, 26, 35, 37, 39, 41, 36, 38, 40, 49})));
    REQUIRE(U.contains(Perm({0,  40, 24, 8,  48, 32, 16, 37, 21, 12, 45, 29, 20,
                             4,  25, 9,  42, 33, 17, 1,  41, 13, 46, 30, 14, 5,
                             38, 22, 43, 34, 18, 2,  35, 26, 10, 31, 15, 6,  39,
                             23, 7,  47, 19, 3,  36, 27, 11, 44, 28, 49})));
    REQUIRE(U.contains(Perm({49, 6,  3,  2,  5,  4,  1,  7,  31, 22, 41, 36, 27,
                             32, 28, 46, 19, 38, 39, 16, 45, 35, 9,  30, 43, 48,
                             33, 12, 14, 44, 23, 8,  13, 26, 47, 21, 11, 40, 17,
                             18, 37, 10, 42, 24, 29, 20, 15, 34, 25, 0})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "012",
                          "3^3:13 and 3^3.2.A(4) intersection",
                          "[quick][schreier-sims]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<27> S, T, U;
    using Perm = SchreierSims<27>::element_type;
    S.add_generator(Perm({0,  17, 22, 1,  15, 23, 2,  16, 21, 3,  11, 25, 4, 9,
                          26, 5,  10, 24, 6,  14, 19, 7,  12, 20, 8,  13, 18}));
    S.add_generator(
        Perm({1,  2,  0,  4,  5,  3,  7,  8,  6,  10, 11, 9,  13, 14,
              12, 16, 17, 15, 19, 20, 18, 22, 23, 21, 25, 26, 24}));
    T.add_generator(Perm({0,  9, 18, 1,  10, 19, 2,  11, 20, 3,  12, 21, 4, 13,
                          22, 5, 14, 23, 6,  15, 24, 7,  16, 25, 8,  17, 26}));
    T.add_generator(
        Perm({0,  2,  1,  6,  8,  7,  3,  5,  4,  9,  11, 10, 15, 17,
              16, 12, 14, 13, 18, 20, 19, 24, 26, 25, 21, 23, 22}));
    T.add_generator(
        Perm({0,  1,  2,  6,  7,  8,  3,  4,  5,  9,  10, 11, 15, 16,
              17, 12, 13, 14, 18, 19, 20, 24, 25, 26, 21, 22, 23}));
    T.add_generator(
        Perm({1,  2,  0,  4,  5,  3,  7,  8,  6,  10, 11, 9,  13, 14,
              12, 16, 17, 15, 19, 20, 18, 22, 23, 21, 25, 26, 24}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 27);
    REQUIRE(
        U.contains(Perm({1,  2,  0,  4,  5,  3,  7,  8,  6,  10, 11, 9,  13, 14,
                         12, 16, 17, 15, 19, 20, 18, 22, 23, 21, 25, 26, 24})));
    REQUIRE(
        U.contains(Perm({3,  4, 5,  6,  7,  8,  0,  1,  2,  12, 13, 14, 15, 16,
                         17, 9, 10, 11, 21, 22, 23, 24, 25, 26, 18, 19, 20})));
    REQUIRE(
        U.contains(Perm({9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22,
                         23, 24, 25, 26, 0,  1,  2,  3,  4,  5,  6,  7,  8})));
  }

  LIBSEMIGROUPS_TEST_CASE(
      "SchreierSimsHelper",
      "013",
      "PGamma(2, 9) wreath Sym(2) and Alt(6)^2.D_8 intersection",
      "[standard][schreier-sims][no-valgrind]") {
    auto              rg = ReportGuard(REPORT);
    SchreierSims<100> S, T, U;
    using Perm = SchreierSims<100>::element_type;
    S.add_generator(Perm(
        {13, 17, 16, 19, 18, 12, 14, 10, 15, 11, 83, 87, 86, 89, 88, 82, 84,
         80, 85, 81, 63, 67, 66, 69, 68, 62, 64, 60, 65, 61, 53, 57, 56, 59,
         58, 52, 54, 50, 55, 51, 33, 37, 36, 39, 38, 32, 34, 30, 35, 31, 3,
         7,  6,  9,  8,  2,  4,  0,  5,  1,  43, 47, 46, 49, 48, 42, 44, 40,
         45, 41, 93, 97, 96, 99, 98, 92, 94, 90, 95, 91, 23, 27, 26, 29, 28,
         22, 24, 20, 25, 21, 73, 77, 76, 79, 78, 72, 74, 70, 75, 71}));
    S.add_generator(Perm(
        {65, 25, 5,  55, 35, 95, 15, 85, 75, 45, 63, 23, 3,  53, 33, 93, 13,
         83, 73, 43, 64, 24, 4,  54, 34, 94, 14, 84, 74, 44, 68, 28, 8,  58,
         38, 98, 18, 88, 78, 48, 61, 21, 1,  51, 31, 91, 11, 81, 71, 41, 60,
         20, 0,  50, 30, 90, 10, 80, 70, 40, 69, 29, 9,  59, 39, 99, 19, 89,
         79, 49, 62, 22, 2,  52, 32, 92, 12, 82, 72, 42, 66, 26, 6,  56, 36,
         96, 16, 86, 76, 46, 67, 27, 7,  57, 37, 97, 17, 87, 77, 47}));
    S.add_generator(Perm(
        {18, 68, 58, 98, 38, 28, 78, 88, 48, 8,  19, 69, 59, 99, 39, 29, 79,
         89, 49, 9,  16, 66, 56, 96, 36, 26, 76, 86, 46, 6,  17, 67, 57, 97,
         37, 27, 77, 87, 47, 7,  13, 63, 53, 93, 33, 23, 73, 83, 43, 3,  11,
         61, 51, 91, 31, 21, 71, 81, 41, 1,  15, 65, 55, 95, 35, 25, 75, 85,
         45, 5,  10, 60, 50, 90, 30, 20, 70, 80, 40, 0,  14, 64, 54, 94, 34,
         24, 74, 84, 44, 4,  12, 62, 52, 92, 32, 22, 72, 82, 42, 2}));
    T.add_generator(Perm(
        {78, 76, 77, 79, 75, 74, 71, 72, 70, 73, 38, 36, 37, 39, 35, 34, 31,
         32, 30, 33, 28, 26, 27, 29, 25, 24, 21, 22, 20, 23, 88, 86, 87, 89,
         85, 84, 81, 82, 80, 83, 8,  6,  7,  9,  5,  4,  1,  2,  0,  3,  48,
         46, 47, 49, 45, 44, 41, 42, 40, 43, 68, 66, 67, 69, 65, 64, 61, 62,
         60, 63, 58, 56, 57, 59, 55, 54, 51, 52, 50, 53, 98, 96, 97, 99, 95,
         94, 91, 92, 90, 93, 18, 16, 17, 19, 15, 14, 11, 12, 10, 13}));
    T.add_generator(Perm(
        {24, 74, 44, 64, 94, 4,  84, 34, 14, 54, 23, 73, 43, 63, 93, 3,  83,
         33, 13, 53, 28, 78, 48, 68, 98, 8,  88, 38, 18, 58, 29, 79, 49, 69,
         99, 9,  89, 39, 19, 59, 25, 75, 45, 65, 95, 5,  85, 35, 15, 55, 22,
         72, 42, 62, 92, 2,  82, 32, 12, 52, 20, 70, 40, 60, 90, 0,  80, 30,
         10, 50, 27, 77, 47, 67, 97, 7,  87, 37, 17, 57, 26, 76, 46, 66, 96,
         6,  86, 36, 16, 56, 21, 71, 41, 61, 91, 1,  81, 31, 11, 51}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 2);
    REQUIRE(U.contains(Perm(
        {11, 1,  91, 81, 71, 61, 51, 41, 31, 21, 10, 0,  90, 80, 70, 60, 50,
         40, 30, 20, 19, 9,  99, 89, 79, 69, 59, 49, 39, 29, 18, 8,  98, 88,
         78, 68, 58, 48, 38, 28, 17, 7,  97, 87, 77, 67, 57, 47, 37, 27, 16,
         6,  96, 86, 76, 66, 56, 46, 36, 26, 15, 5,  95, 85, 75, 65, 55, 45,
         35, 25, 14, 4,  94, 84, 74, 64, 54, 44, 34, 24, 13, 3,  93, 83, 73,
         63, 53, 43, 33, 23, 12, 2,  92, 82, 72, 62, 52, 42, 32, 22})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "014",
                          "Alt(6)^2.2^2:4 and Alt(6)^2.4 intersection",
                          "[standard][schreier-sims][no-valgrind]") {
    auto              rg = ReportGuard(REPORT);
    SchreierSims<100> S, T, U;
    using Perm = SchreierSims<100>::element_type;
    S.add_generator(Perm(
        {68, 62, 69, 67, 66, 64, 61, 65, 60, 63, 98, 92, 99, 97, 96, 94, 91,
         95, 90, 93, 38, 32, 39, 37, 36, 34, 31, 35, 30, 33, 78, 72, 79, 77,
         76, 74, 71, 75, 70, 73, 8,  2,  9,  7,  6,  4,  1,  5,  0,  3,  28,
         22, 29, 27, 26, 24, 21, 25, 20, 23, 18, 12, 19, 17, 16, 14, 11, 15,
         10, 13, 88, 82, 89, 87, 86, 84, 81, 85, 80, 83, 58, 52, 59, 57, 56,
         54, 51, 55, 50, 53, 48, 42, 49, 47, 46, 44, 41, 45, 40, 43}));
    S.add_generator(Perm(
        {72, 92, 12, 82, 32, 62, 42, 22, 52, 2,  70, 90, 10, 80, 30, 60, 40,
         20, 50, 0,  75, 95, 15, 85, 35, 65, 45, 25, 55, 5,  76, 96, 16, 86,
         36, 66, 46, 26, 56, 6,  71, 91, 11, 81, 31, 61, 41, 21, 51, 1,  79,
         99, 19, 89, 39, 69, 49, 29, 59, 9,  78, 98, 18, 88, 38, 68, 48, 28,
         58, 8,  74, 94, 14, 84, 34, 64, 44, 24, 54, 4,  77, 97, 17, 87, 37,
         67, 47, 27, 57, 7,  73, 93, 13, 83, 33, 63, 43, 23, 53, 3}));
    T.add_generator(Perm(
        {20, 26, 23, 28, 24, 29, 25, 22, 27, 21, 40, 46, 43, 48, 44, 49, 45,
         42, 47, 41, 60, 66, 63, 68, 64, 69, 65, 62, 67, 61, 70, 76, 73, 78,
         74, 79, 75, 72, 77, 71, 50, 56, 53, 58, 54, 59, 55, 52, 57, 51, 30,
         36, 33, 38, 34, 39, 35, 32, 37, 31, 90, 96, 93, 98, 94, 99, 95, 92,
         97, 91, 10, 16, 13, 18, 14, 19, 15, 12, 17, 11, 0,  6,  3,  8,  4,
         9,  5,  2,  7,  1,  80, 86, 83, 88, 84, 89, 85, 82, 87, 81}));
    T.add_generator(Perm(
        {21, 11, 71, 51, 41, 91, 1,  31, 61, 81, 24, 14, 74, 54, 44, 94, 4,
         34, 64, 84, 27, 17, 77, 57, 47, 97, 7,  37, 67, 87, 28, 18, 78, 58,
         48, 98, 8,  38, 68, 88, 25, 15, 75, 55, 45, 95, 5,  35, 65, 85, 20,
         10, 70, 50, 40, 90, 0,  30, 60, 80, 26, 16, 76, 56, 46, 96, 6,  36,
         66, 86, 23, 13, 73, 53, 43, 93, 3,  33, 63, 83, 22, 12, 72, 52, 42,
         92, 2,  32, 62, 82, 29, 19, 79, 59, 49, 99, 9,  39, 69, 89}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 4);
    REQUIRE(U.contains(Perm(
        {1,  11, 21, 31, 41, 51, 61, 71, 81, 91, 0,  10, 20, 30, 40, 50, 60,
         70, 80, 90, 9,  19, 29, 39, 49, 59, 69, 79, 89, 99, 8,  18, 28, 38,
         48, 58, 68, 78, 88, 98, 7,  17, 27, 37, 47, 57, 67, 77, 87, 97, 6,
         16, 26, 36, 46, 56, 66, 76, 86, 96, 5,  15, 25, 35, 45, 55, 65, 75,
         85, 95, 4,  14, 24, 34, 44, 54, 64, 74, 84, 94, 3,  13, 23, 33, 43,
         53, 63, 73, 83, 93, 2,  12, 22, 32, 42, 52, 62, 72, 82, 92})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "015",
                          "3^3(S(4) x 2) and ASL(3, 3) intersection",
                          "[quick][schreier-sims]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<27> S, T, U;
    using Perm = SchreierSims<27>::element_type;
    S.add_generator(Perm({0, 6,  3,  25, 22, 19, 14, 11, 17, 9,  15, 12, 7, 4,
                          1, 23, 20, 26, 18, 24, 21, 16, 13, 10, 5,  2,  8}));
    S.add_generator(Perm({0, 18, 9, 25, 16, 7,  14, 5,  23, 3, 21, 12, 19, 10,
                          1, 17, 8, 26, 6,  24, 15, 22, 13, 4, 11, 2,  20}));
    S.add_generator(Perm({0, 9,  18, 14, 23, 5,  25, 7,  16, 1, 10, 19, 12, 21,
                          3, 26, 8,  17, 2,  11, 20, 13, 22, 4, 24, 6,  15}));
    S.add_generator(
        Perm({1,  2,  0,  4,  5,  3,  7,  8,  6,  10, 11, 9,  13, 14,
              12, 16, 17, 15, 19, 20, 18, 22, 23, 21, 25, 26, 24}));
    T.add_generator(Perm({0,  26, 13, 1,  24, 14, 2, 25, 12, 9, 8,  22, 10, 6,
                          23, 11, 7,  21, 18, 17, 4, 19, 15, 5, 20, 16, 3}));
    T.add_generator(Perm({0,  17, 22, 1,  15, 23, 2,  16, 21, 3,  11, 25, 4, 9,
                          26, 5,  10, 24, 6,  14, 19, 7,  12, 20, 8,  13, 18}));
    T.add_generator(
        Perm({1,  2,  0,  4,  5,  3,  7,  8,  6,  10, 11, 9,  13, 14,
              12, 16, 17, 15, 19, 20, 18, 22, 23, 21, 25, 26, 24}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 648);
    REQUIRE(
        U.contains(Perm({0,  9, 18, 6,  15, 24, 3,  12, 21, 1,  10, 19, 7, 16,
                         25, 4, 13, 22, 2,  11, 20, 8,  17, 26, 5,  14, 23})));
    REQUIRE(
        U.contains(Perm({0,  1,  2,  25, 26, 24, 14, 12, 13, 3,  4,  5, 19, 20,
                         18, 17, 15, 16, 6,  7,  8,  22, 23, 21, 11, 9, 10})));
    REQUIRE(
        U.contains(Perm({1,  2,  0,  26, 24, 25, 12, 13, 14, 4,  5, 3,  20, 18,
                         19, 15, 16, 17, 7,  8,  6,  23, 21, 22, 9, 10, 11})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "016",
                          "7^2:3 x Q(8) and 7^2:D(2*6) intersection",
                          "[quick][schreier-sims]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<49> S, T, U;
    using Perm = SchreierSims<49>::element_type;
    S.add_generator(Perm({0,  28, 35, 42, 7,  14, 21, 1,  29, 36, 43, 8,  15,
                          22, 2,  30, 37, 44, 9,  16, 23, 3,  31, 38, 45, 10,
                          17, 24, 4,  32, 39, 46, 11, 18, 25, 5,  33, 40, 47,
                          12, 19, 26, 6,  34, 41, 48, 13, 20, 27}));
    S.add_generator(
        Perm({0, 44, 10, 18, 26, 34, 36, 41, 22, 42, 5,  31, 16, 11, 43, 19, 30,
              7, 6,  39, 24, 9,  32, 27, 38, 14, 1,  47, 17, 13, 40, 29, 46, 21,
              2, 25, 3,  15, 48, 37, 12, 28, 33, 35, 4,  23, 8,  45, 20}));
    S.add_generator(Perm({0,  3,  4,  5,  6,  1,  2,  21, 24, 25, 26, 27, 22,
                          23, 28, 31, 32, 33, 34, 29, 30, 35, 38, 39, 40, 41,
                          36, 37, 42, 45, 46, 47, 48, 43, 44, 7,  10, 11, 12,
                          13, 8,  9,  14, 17, 18, 19, 20, 15, 16}));
    S.add_generator(Perm({1,  3,  5,  2,  0,  6,  4,  8,  10, 12, 9,  7,  13,
                          11, 15, 17, 19, 16, 14, 20, 18, 22, 24, 26, 23, 21,
                          27, 25, 29, 31, 33, 30, 28, 34, 32, 36, 38, 40, 37,
                          35, 41, 39, 43, 45, 47, 44, 42, 48, 46}));
    T.add_generator(Perm({0,  32, 40, 48, 8,  16, 24, 7,  4,  47, 34, 22, 37,
                          17, 14, 25, 5,  13, 36, 30, 45, 21, 11, 33, 6,  15,
                          44, 38, 28, 46, 19, 41, 1,  23, 10, 35, 18, 12, 27,
                          43, 2,  31, 42, 39, 26, 20, 29, 9,  3}));
    T.add_generator(Perm({0,  32, 40, 48, 8,  16, 24, 1,  28, 41, 46, 10, 19,
                          23, 2,  31, 35, 43, 12, 18, 27, 3,  29, 39, 42, 9,
                          20, 26, 4,  34, 37, 47, 7,  17, 22, 5,  30, 36, 45,
                          13, 14, 25, 6,  33, 38, 44, 11, 15, 21}));
    T.add_generator(Perm({0,  4,  5,  6,  1,  2,  3,  28, 32, 33, 34, 29, 30,
                          31, 35, 39, 40, 41, 36, 37, 38, 42, 46, 47, 48, 43,
                          44, 45, 7,  11, 12, 13, 8,  9,  10, 14, 18, 19, 20,
                          15, 16, 17, 21, 25, 26, 27, 22, 23, 24}));
    T.add_generator(Perm({1,  3,  5,  2,  0,  6,  4,  8,  10, 12, 9,  7,  13,
                          11, 15, 17, 19, 16, 14, 20, 18, 22, 24, 26, 23, 21,
                          27, 25, 29, 31, 33, 30, 28, 34, 32, 36, 38, 40, 37,
                          35, 41, 39, 43, 45, 47, 44, 42, 48, 46}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 98);
    REQUIRE(U.contains(Perm({0,  4,  5,  6,  1,  2,  3,  28, 32, 33, 34, 29, 30,
                             31, 35, 39, 40, 41, 36, 37, 38, 42, 46, 47, 48, 43,
                             44, 45, 7,  11, 12, 13, 8,  9,  10, 14, 18, 19, 20,
                             15, 16, 17, 21, 25, 26, 27, 22, 23, 24})));
    REQUIRE(U.contains(Perm({1,  0,  6,  4,  3,  5,  2,  29, 28, 34, 32, 31, 33,
                             30, 36, 35, 41, 39, 38, 40, 37, 43, 42, 48, 46, 45,
                             47, 44, 8,  7,  13, 11, 10, 12, 9,  15, 14, 20, 18,
                             17, 19, 16, 22, 21, 27, 25, 24, 26, 23})));
    REQUIRE(U.contains(Perm({7,  11, 12, 13, 8,  9,  10, 0,  4,  5,  6,  1,  2,
                             3,  42, 46, 47, 48, 43, 44, 45, 28, 32, 33, 34, 29,
                             30, 31, 21, 25, 26, 27, 22, 23, 24, 35, 39, 40, 41,
                             36, 37, 38, 14, 18, 19, 20, 15, 16, 17})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "018",
                          "D(2*53) and 53:13 intersection",
                          "[quick][schreier-sims]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<53> S, T, U;
    using Perm = SchreierSims<53>::element_type;
    S.add_generator(Perm(
        {0,  27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43,
         44, 45, 46, 47, 48, 49, 50, 51, 52, 1,  2,  3,  4,  5,  6,  7,  8,  9,
         10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26}));
    S.add_generator(Perm(
        {1,  2,  18, 48, 35, 11, 24, 20, 40, 30, 37, 36, 10, 5,  34, 4,  13, 47,
         3,  15, 25, 43, 29, 9,  12, 16, 52, 0,  27, 44, 41, 39, 8,  23, 6,  49,
         38, 31, 50, 42, 21, 45, 51, 26, 28, 22, 33, 14, 19, 7,  32, 46, 17}));
    T.add_generator(Perm(
        {0,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21,
         22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39,
         40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 1,  2,  3,  4}));
    T.add_generator(Perm(
        {1,  2,  18, 48, 35, 11, 24, 20, 40, 30, 37, 36, 10, 5,  34, 4,  13, 47,
         3,  15, 25, 43, 29, 9,  12, 16, 52, 0,  27, 44, 41, 39, 8,  23, 6,  49,
         38, 31, 50, 42, 21, 45, 51, 26, 28, 22, 33, 14, 19, 7,  32, 46, 17}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 53);
    REQUIRE(U.contains(Perm(
        {1,  2,  18, 48, 35, 11, 24, 20, 40, 30, 37, 36, 10, 5,  34, 4,  13, 47,
         3,  15, 25, 43, 29, 9,  12, 16, 52, 0,  27, 44, 41, 39, 8,  23, 6,  49,
         38, 31, 50, 42, 21, 45, 51, 26, 28, 22, 33, 14, 19, 7,  32, 46, 17})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "019",
                          "2^6:(7 x D_14) and PSL(2, 6)^2.4 intersection",
                          "[quick][schreier-sims]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<64> S, T, U;
    using Perm = SchreierSims<64>::element_type;
    S.add_generator(
        Perm({0,  32, 5,  37, 2,  34, 7,  39, 20, 52, 17, 49, 22, 54, 19, 51,
              8,  40, 13, 45, 10, 42, 15, 47, 28, 60, 25, 57, 30, 62, 27, 59,
              16, 48, 21, 53, 18, 50, 23, 55, 4,  36, 1,  33, 6,  38, 3,  35,
              24, 56, 29, 61, 26, 58, 31, 63, 12, 44, 9,  41, 14, 46, 11, 43}));
    S.add_generator(
        Perm({0,  17, 2,  19, 1,  16, 3,  18, 8,  25, 10, 27, 9,  24, 11, 26,
              4,  21, 6,  23, 5,  20, 7,  22, 12, 29, 14, 31, 13, 28, 15, 30,
              32, 49, 34, 51, 33, 48, 35, 50, 40, 57, 42, 59, 41, 56, 43, 58,
              36, 53, 38, 55, 37, 52, 39, 54, 44, 61, 46, 63, 45, 60, 47, 62}));
    S.add_generator(
        Perm({1,  0,  3,  2,  5,  4,  7,  6,  9,  8,  11, 10, 13, 12, 15, 14,
              17, 16, 19, 18, 21, 20, 23, 22, 25, 24, 27, 26, 29, 28, 31, 30,
              33, 32, 35, 34, 37, 36, 39, 38, 41, 40, 43, 42, 45, 44, 47, 46,
              49, 48, 51, 50, 53, 52, 55, 54, 57, 56, 59, 58, 61, 60, 63, 62}));
    T.add_generator(
        Perm({10, 34, 58, 42, 50, 2, 18, 26, 9,  33, 57, 41, 49, 1, 17, 25,
              15, 39, 63, 47, 55, 7, 23, 31, 13, 37, 61, 45, 53, 5, 21, 29,
              12, 36, 60, 44, 52, 4, 20, 28, 14, 38, 62, 46, 54, 6, 22, 30,
              11, 35, 59, 43, 51, 3, 19, 27, 8,  32, 56, 40, 48, 0, 16, 24}));
    T.add_generator(
        Perm({39, 23, 63, 55, 31, 15, 47, 7, 37, 21, 61, 53, 29, 13, 45, 5,
              32, 16, 56, 48, 24, 8,  40, 0, 35, 19, 59, 51, 27, 11, 43, 3,
              33, 17, 57, 49, 25, 9,  41, 1, 34, 18, 58, 50, 26, 10, 42, 2,
              36, 20, 60, 52, 28, 12, 44, 4, 38, 22, 62, 54, 30, 14, 46, 6}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.contains(Perm({3,  2,  1,  0,  7,  6,  5,  4,  11, 10, 9,  8,  15,
                             14, 13, 12, 19, 18, 17, 16, 23, 22, 21, 20, 27, 26,
                             25, 24, 31, 30, 29, 28, 35, 34, 33, 32, 39, 38, 37,
                             36, 43, 42, 41, 40, 47, 46, 45, 44, 51, 50, 49, 48,
                             55, 54, 53, 52, 59, 58, 57, 56, 63, 62, 61, 60})));
    REQUIRE(U.contains(Perm({4,  5,  6,  7,  0,  1,  2,  3,  12, 13, 14, 15, 8,
                             9,  10, 11, 20, 21, 22, 23, 16, 17, 18, 19, 28, 29,
                             30, 31, 24, 25, 26, 27, 36, 37, 38, 39, 32, 33, 34,
                             35, 44, 45, 46, 47, 40, 41, 42, 43, 52, 53, 54, 55,
                             48, 49, 50, 51, 60, 61, 62, 63, 56, 57, 58, 59})));
    REQUIRE(U.contains(Perm({24, 25, 26, 27, 28, 29, 30, 31, 16, 17, 18, 19, 20,
                             21, 22, 23, 8,  9,  10, 11, 12, 13, 14, 15, 0,  1,
                             2,  3,  4,  5,  6,  7,  56, 57, 58, 59, 60, 61, 62,
                             63, 48, 49, 50, 51, 52, 53, 54, 55, 40, 41, 42, 43,
                             44, 45, 46, 47, 32, 33, 34, 35, 36, 37, 38, 39})));
    REQUIRE(U.contains(Perm({32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44,
                             45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57,
                             58, 59, 60, 61, 62, 63, 0,  1,  2,  3,  4,  5,  6,
                             7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
                             20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31})));
    REQUIRE(U.size() == 16);
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "020",
                          "2^6:(S_3 x GL(3, 2)) and 2.6:Alt(7) intersection",
                          "[quick][schreier-sims]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<64> S, T, U;
    using Perm = SchreierSims<64>::element_type;
    S.add_generator(
        Perm({0,  45, 9,  36, 31, 50, 22, 59, 2,  47, 11, 38, 29, 48, 20, 57,
              8,  37, 1,  44, 23, 58, 30, 51, 10, 39, 3,  46, 21, 56, 28, 49,
              16, 61, 25, 52, 15, 34, 6,  43, 18, 63, 27, 54, 13, 32, 4,  41,
              24, 53, 17, 60, 7,  42, 14, 35, 26, 55, 19, 62, 5,  40, 12, 33}));
    S.add_generator(
        Perm({0,  9,  49, 56, 2,  11, 51, 58, 1,  8,  48, 57, 3,  10, 50, 59,
              4,  13, 53, 60, 6,  15, 55, 62, 5,  12, 52, 61, 7,  14, 54, 63,
              32, 41, 17, 24, 34, 43, 19, 26, 33, 40, 16, 25, 35, 42, 18, 27,
              36, 45, 21, 28, 38, 47, 23, 30, 37, 44, 20, 29, 39, 46, 22, 31}));
    S.add_generator(
        Perm({1,  0,  3,  2,  5,  4,  7,  6,  9,  8,  11, 10, 13, 12, 15, 14,
              17, 16, 19, 18, 21, 20, 23, 22, 25, 24, 27, 26, 29, 28, 31, 30,
              33, 32, 35, 34, 37, 36, 39, 38, 41, 40, 43, 42, 45, 44, 47, 46,
              49, 48, 51, 50, 53, 52, 55, 54, 57, 56, 59, 58, 61, 60, 63, 62}));
    T.add_generator(
        Perm({0,  45, 58, 23, 28, 49, 38, 11, 2,  47, 56, 21, 30, 51, 36, 9,
              8,  37, 50, 31, 20, 57, 46, 3,  10, 39, 48, 29, 22, 59, 44, 1,
              32, 13, 26, 55, 60, 17, 6,  43, 34, 15, 24, 53, 62, 19, 4,  41,
              40, 5,  18, 63, 52, 25, 14, 35, 42, 7,  16, 61, 54, 27, 12, 33}));
    T.add_generator(
        Perm({0,  30, 53, 43, 38, 56, 19, 13, 1,  31, 52, 42, 39, 57, 18, 12,
              4,  26, 49, 47, 34, 60, 23, 9,  5,  27, 48, 46, 35, 61, 22, 8,
              16, 14, 37, 59, 54, 40, 3,  29, 17, 15, 36, 58, 55, 41, 2,  28,
              20, 10, 33, 63, 50, 44, 7,  25, 21, 11, 32, 62, 51, 45, 6,  24}));
    T.add_generator(
        Perm({1,  0,  3,  2,  5,  4,  7,  6,  9,  8,  11, 10, 13, 12, 15, 14,
              17, 16, 19, 18, 21, 20, 23, 22, 25, 24, 27, 26, 29, 28, 31, 30,
              33, 32, 35, 34, 37, 36, 39, 38, 41, 40, 43, 42, 45, 44, 47, 46,
              49, 48, 51, 50, 53, 52, 55, 54, 57, 56, 59, 58, 61, 60, 63, 62}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 64);
    REQUIRE(U.contains(Perm({1,  0,  3,  2,  5,  4,  7,  6,  9,  8,  11, 10, 13,
                             12, 15, 14, 17, 16, 19, 18, 21, 20, 23, 22, 25, 24,
                             27, 26, 29, 28, 31, 30, 33, 32, 35, 34, 37, 36, 39,
                             38, 41, 40, 43, 42, 45, 44, 47, 46, 49, 48, 51, 50,
                             53, 52, 55, 54, 57, 56, 59, 58, 61, 60, 63, 62})));
    REQUIRE(U.contains(Perm({2,  3,  0,  1,  6,  7,  4,  5,  10, 11, 8,  9,  14,
                             15, 12, 13, 18, 19, 16, 17, 22, 23, 20, 21, 26, 27,
                             24, 25, 30, 31, 28, 29, 34, 35, 32, 33, 38, 39, 36,
                             37, 42, 43, 40, 41, 46, 47, 44, 45, 50, 51, 48, 49,
                             54, 55, 52, 53, 58, 59, 56, 57, 62, 63, 60, 61})));
    REQUIRE(U.contains(Perm({4,  5,  6,  7,  0,  1,  2,  3,  12, 13, 14, 15, 8,
                             9,  10, 11, 20, 21, 22, 23, 16, 17, 18, 19, 28, 29,
                             30, 31, 24, 25, 26, 27, 36, 37, 38, 39, 32, 33, 34,
                             35, 44, 45, 46, 47, 40, 41, 42, 43, 52, 53, 54, 55,
                             48, 49, 50, 51, 60, 61, 62, 63, 56, 57, 58, 59})));
    REQUIRE(U.contains(Perm({8,  9,  10, 11, 12, 13, 14, 15, 0,  1,  2,  3,  4,
                             5,  6,  7,  24, 25, 26, 27, 28, 29, 30, 31, 16, 17,
                             18, 19, 20, 21, 22, 23, 40, 41, 42, 43, 44, 45, 46,
                             47, 32, 33, 34, 35, 36, 37, 38, 39, 56, 57, 58, 59,
                             60, 61, 62, 63, 48, 49, 50, 51, 52, 53, 54, 55})));
    REQUIRE(U.contains(Perm({16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28,
                             29, 30, 31, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,
                             10, 11, 12, 13, 14, 15, 48, 49, 50, 51, 52, 53, 54,
                             55, 56, 57, 58, 59, 60, 61, 62, 63, 32, 33, 34, 35,
                             36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47})));
    REQUIRE(U.contains(Perm({32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44,
                             45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57,
                             58, 59, 60, 61, 62, 63, 0,  1,  2,  3,  4,  5,  6,
                             7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
                             20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "021",
                          "AGL(7, 2) and PGL(2, 127) intersection",
                          "[extreme][schreier-sims][no-valgrind]") {
    auto              rg = ReportGuard(REPORT);
    SchreierSims<128> S, T, U;
    using Perm = SchreierSims<128>::element_type;
    S.add_generator(
        Perm({0,   34,  31,  61,  1,   35,  30,  60,  96,  66, 127, 93,  97,
              67,  126, 92,  4,   38,  27,  57,  5,   39,  26, 56,  100, 70,
              123, 89,  101, 71,  122, 88,  16,  50,  15,  45, 17,  51,  14,
              44,  112, 82,  111, 77,  113, 83,  110, 76,  20, 54,  11,  41,
              21,  55,  10,  40,  116, 86,  107, 73,  117, 87, 106, 72,  32,
              2,   63,  29,  33,  3,   62,  28,  64,  98,  95, 125, 65,  99,
              94,  124, 36,  6,   59,  25,  37,  7,   58,  24, 68,  102, 91,
              121, 69,  103, 90,  120, 48,  18,  47,  13,  49, 19,  46,  12,
              80,  114, 79,  109, 81,  115, 78,  108, 52,  22, 43,  9,   53,
              23,  42,  8,   84,  118, 75,  105, 85,  119, 74, 104}));
    S.add_generator(
        Perm({0,   49,  78,  127, 13,  60,  67,  114, 2,   51,  76,  125, 15,
              62,  65,  112, 20,  37,  90,  107, 25,  40,  87,  102, 22,  39,
              88,  105, 27,  42,  85,  100, 8,   57,  70,  119, 5,   52,  75,
              122, 10,  59,  68,  117, 7,   54,  73,  120, 28,  45,  82,  99,
              17,  32,  95,  110, 30,  47,  80,  97,  19,  34,  93,  108, 64,
              113, 14,  63,  77,  124, 3,   50,  66,  115, 12,  61,  79,  126,
              1,   48,  84,  101, 26,  43,  89,  104, 23,  38,  86,  103, 24,
              41,  91,  106, 21,  36,  72,  121, 6,   55,  69,  116, 11,  58,
              74,  123, 4,   53,  71,  118, 9,   56,  92,  109, 18,  35,  81,
              96,  31,  46,  94,  111, 16,  33,  83,  98,  29,  44}));
    S.add_generator(
        Perm({1,   0,   3,   2,   5,   4,   7,   6,   9,   8,   11,  10,  13,
              12,  15,  14,  17,  16,  19,  18,  21,  20,  23,  22,  25,  24,
              27,  26,  29,  28,  31,  30,  33,  32,  35,  34,  37,  36,  39,
              38,  41,  40,  43,  42,  45,  44,  47,  46,  49,  48,  51,  50,
              53,  52,  55,  54,  57,  56,  59,  58,  61,  60,  63,  62,  65,
              64,  67,  66,  69,  68,  71,  70,  73,  72,  75,  74,  77,  76,
              79,  78,  81,  80,  83,  82,  85,  84,  87,  86,  89,  88,  91,
              90,  93,  92,  95,  94,  97,  96,  99,  98,  101, 100, 103, 102,
              105, 104, 107, 106, 109, 108, 111, 110, 113, 112, 115, 114, 117,
              116, 119, 118, 121, 120, 123, 122, 125, 124, 127, 126}));
    T.add_generator(
        Perm({0,   1,   8,   65,  42,  52,  11,  63,  35,  77,  38,  54,  70,
              31,  73,  45,  97,  56,  66,  43,  100, 107, 108, 88,  84,  33,
              12,  76,  61,  109, 21,  62,  113, 5,   68,  80,  93,  48,  39,
              22,  27,  106, 9,   18,  37,  105, 78,  67,  19,  94,  57,  83,
              120, 102, 4,   50,  115, 59,  23,  117, 17,  95,  82,  111, 126,
              53,  122, 98,  101, 16,  72,  34,  79,  112, 44,  118, 64,  124,
              29,  74,  36,  92,  7,   10,  114, 55,  110, 125, 26,  13,  91,
              119, 85,  49,  121, 58,  104, 6,   116, 25,  51,  14,  89,  41,
              30,  3,   71,  90,  99,  2,   81,  86,  69,  60,  15,  28,  40,
              103, 123, 46,  87,  127, 47,  32,  96,  75,  24,  20}));
    T.add_generator(Perm(
        {82,  0,   72,  58,  33,  107, 21,  120, 46, 115, 77,  28,  16, 47,  14,
         61,  75,  48,  25,  8,   29,  31,  71,  20, 49,  73,  13,  22, 113, 23,
         112, 6,   40,  70,  76,  102, 78,  96,  34, 87,  110, 90,  79, 63,  10,
         15,  19,  26,  59,  69,  119, 97,  116, 64, 109, 121, 5,   50, 126, 17,
         123, 45,  55,  88,  95,  99,  104, 51,  35, 24,  4,   27,  80, 18,  36,
         12,  38,  44,  74,  108, 2,   91,  1,   92, 101, 41,  65,  94, 43,  83,
         85,  106, 89,  66,  39,  53,  100, 67,  98, 86,  37,  127, 68, 122, 93,
         7,   81,  56,  42,  114, 32,  30,  111, 11, 54,  118, 117, 52, 9,   57,
         105, 62,  125, 124, 60,  103, 3,   84}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 8);
    REQUIRE(U.contains(
        Perm({31,  30,  29,  28,  27,  26,  25,  24,  23,  22,  21,  20,  19,
              18,  17,  16,  15,  14,  13,  12,  11,  10,  9,   8,   7,   6,
              5,   4,   3,   2,   1,   0,   127, 126, 125, 124, 123, 122, 121,
              120, 119, 118, 117, 116, 115, 114, 113, 112, 111, 110, 109, 108,
              107, 106, 105, 104, 103, 102, 101, 100, 99,  98,  97,  96,  95,
              94,  93,  92,  91,  90,  89,  88,  87,  86,  85,  84,  83,  82,
              81,  80,  79,  78,  77,  76,  75,  74,  73,  72,  71,  70,  69,
              68,  67,  66,  65,  64,  63,  62,  61,  60,  59,  58,  57,  56,
              55,  54,  53,  52,  51,  50,  49,  48,  47,  46,  45,  44,  43,
              42,  41,  40,  39,  38,  37,  36,  35,  34,  33,  32})));
    REQUIRE(U.contains(
        Perm({32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,  44,
              45,  46,  47,  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,
              58,  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,  69,  70,
              71,  72,  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,  83,
              84,  85,  86,  87,  88,  89,  90,  91,  92,  93,  94,  95,  96,
              97,  98,  99,  100, 101, 102, 103, 104, 105, 106, 107, 108, 109,
              110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122,
              123, 124, 125, 126, 127, 0,   1,   2,   3,   4,   5,   6,   7,
              8,   9,   10,  11,  12,  13,  14,  15,  16,  17,  18,  19,  20,
              21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "022",
                          "PSL(2, 127) and AGL(1, 2^7) intersection",
                          "[quick][schreier-sims]") {
    auto              rg = ReportGuard(REPORT);
    SchreierSims<128> S, T, U;
    using Perm = SchreierSims<128>::element_type;
    S.add_generator(Perm(
        {0,   1,   35,  53,  9,   120, 54,  111, 80, 124, 39, 4,  72,  62,  112,
         105, 6,   115, 122, 18,  51,  90,  99,  26, 114, 5,  70, 64,  95,  2,
         107, 82,  60,  52,  101, 36,  49,  19,  22, 108, 76, 71, 77,  66,  48,
         3,   29,  98,  43,  121, 59,  10,  87,  89, 42,  57, 28, 117, 88,  103,
         56,  58,  7,   86,  24,  102, 47,  116, 14, 97,  79, 68, 74,  69,  37,
         123, 126, 96,  109, 44,  93,  85,  63,  38, 15,  50, 81, 75,  12,  31,
         119, 46,  55,  94,  127, 23,  30,  11,  40, 33,  83, 73, 13,  106, 21,
         65,  34,  91,  25,  8,   92,  110, 16,  17, 45,  61, 27, 41,  32,  78,
         125, 20,  67,  113, 104, 118, 84,  100}));
    S.add_generator(Perm(
        {82,  0,   72,  58,  33,  107, 21,  120, 46, 115, 77,  28,  16, 47,  14,
         61,  75,  48,  25,  8,   29,  31,  71,  20, 49,  73,  13,  22, 113, 23,
         112, 6,   40,  70,  76,  102, 78,  96,  34, 87,  110, 90,  79, 63,  10,
         15,  19,  26,  59,  69,  119, 97,  116, 64, 109, 121, 5,   50, 126, 17,
         123, 45,  55,  88,  95,  99,  104, 51,  35, 24,  4,   27,  80, 18,  36,
         12,  38,  44,  74,  108, 2,   91,  1,   92, 101, 41,  65,  94, 43,  83,
         85,  106, 89,  66,  39,  53,  100, 67,  98, 86,  37,  127, 68, 122, 93,
         7,   81,  56,  42,  114, 32,  30,  111, 11, 54,  118, 117, 52, 9,   57,
         105, 62,  125, 124, 60,  103, 3,   84}));
    T.add_generator(
        Perm({0,   34,  54,  20,  1,   35,  55,  21,  100, 70,  82, 112, 101,
              71,  83,  113, 4,   38,  50,  16,  5,   39,  51,  17, 96,  66,
              86,  116, 97,  67,  87,  117, 8,   42,  62,  28,  9,  43,  63,
              29,  108, 78,  90,  120, 109, 79,  91,  121, 12,  46, 58,  24,
              13,  47,  59,  25,  104, 74,  94,  124, 105, 75,  95, 125, 32,
              2,   22,  52,  33,  3,   23,  53,  68,  102, 114, 80, 69,  103,
              115, 81,  36,  6,   18,  48,  37,  7,   19,  49,  64, 98,  118,
              84,  65,  99,  119, 85,  40,  10,  30,  60,  41,  11, 31,  61,
              76,  110, 122, 88,  77,  111, 123, 89,  44,  14,  26, 56,  45,
              15,  27,  57,  72,  106, 126, 92,  73,  107, 127, 93}));
    T.add_generator(Perm(
        {0,  59, 120, 67, 54, 13, 78, 117, 1,  58, 121, 66, 55, 12, 79, 116,
         2,  57, 122, 65, 52, 15, 76, 119, 3,  56, 123, 64, 53, 14, 77, 118,
         4,  63, 124, 71, 50, 9,  74, 113, 5,  62, 125, 70, 51, 8,  75, 112,
         6,  61, 126, 69, 48, 11, 72, 115, 7,  60, 127, 68, 49, 10, 73, 114,
         16, 43, 104, 83, 38, 29, 94, 101, 17, 42, 105, 82, 39, 28, 95, 100,
         18, 41, 106, 81, 36, 31, 92, 103, 19, 40, 107, 80, 37, 30, 93, 102,
         20, 47, 108, 87, 34, 25, 90, 97,  21, 46, 109, 86, 35, 24, 91, 96,
         22, 45, 110, 85, 32, 27, 88, 99,  23, 44, 111, 84, 33, 26, 89, 98}));
    T.add_generator(
        Perm({1,   0,   3,   2,   5,   4,   7,   6,   9,   8,   11,  10,  13,
              12,  15,  14,  17,  16,  19,  18,  21,  20,  23,  22,  25,  24,
              27,  26,  29,  28,  31,  30,  33,  32,  35,  34,  37,  36,  39,
              38,  41,  40,  43,  42,  45,  44,  47,  46,  49,  48,  51,  50,
              53,  52,  55,  54,  57,  56,  59,  58,  61,  60,  63,  62,  65,
              64,  67,  66,  69,  68,  71,  70,  73,  72,  75,  74,  77,  76,
              79,  78,  81,  80,  83,  82,  85,  84,  87,  86,  89,  88,  91,
              90,  93,  92,  95,  94,  97,  96,  99,  98,  101, 100, 103, 102,
              105, 104, 107, 106, 109, 108, 111, 110, 113, 112, 115, 114, 117,
              116, 119, 118, 121, 120, 123, 122, 125, 124, 127, 126}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 4);
    REQUIRE(U.contains(
        Perm({63,  62,  61,  60,  59,  58,  57,  56,  55,  54,  53,  52,  51,
              50,  49,  48,  47,  46,  45,  44,  43,  42,  41,  40,  39,  38,
              37,  36,  35,  34,  33,  32,  31,  30,  29,  28,  27,  26,  25,
              24,  23,  22,  21,  20,  19,  18,  17,  16,  15,  14,  13,  12,
              11,  10,  9,   8,   7,   6,   5,   4,   3,   2,   1,   0,   127,
              126, 125, 124, 123, 122, 121, 120, 119, 118, 117, 116, 115, 114,
              113, 112, 111, 110, 109, 108, 107, 106, 105, 104, 103, 102, 101,
              100, 99,  98,  97,  96,  95,  94,  93,  92,  91,  90,  89,  88,
              87,  86,  85,  84,  83,  82,  81,  80,  79,  78,  77,  76,  75,
              74,  73,  72,  71,  70,  69,  68,  67,  66,  65,  64})));
    REQUIRE(U.contains(
        Perm({64,  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,
              77,  78,  79,  80,  81,  82,  83,  84,  85,  86,  87,  88,  89,
              90,  91,  92,  93,  94,  95,  96,  97,  98,  99,  100, 101, 102,
              103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115,
              116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 0,
              1,   2,   3,   4,   5,   6,   7,   8,   9,   10,  11,  12,  13,
              14,  15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,
              27,  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,  38,  39,
              40,  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,  52,
              53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63})));
  }

  LIBSEMIGROUPS_TEST_CASE(
      "SchreierSimsHelper",
      "023",
      "PSL(3, 4).2 and PSL(3, 4).2 (nontrivial) intersection",
      "[quick][schreier-sims]") {
    auto              rg = ReportGuard(REPORT);
    SchreierSims<120> S, T, U;
    using Perm = SchreierSims<120>::element_type;
    S.add_generator(Perm(
        {35,  1,   66,  63,  98,  53,  117, 55,  20,  118, 56, 73,  93,  57,
         112, 15,  32,  119, 42,  19,  8,   25,  23,  22,  24, 21,  72,  75,
         64,  97,  38,  96,  16,  33,  88,  0,   52,  87,  30, 109, 108, 94,
         18,  43,  44,  100, 89,  47,  50,  116, 48,  51,  36, 5,   54,  7,
         10,  13,  76,  59,  85,  61,  62,  3,   28,  101, 2,  104, 80,  113,
         70,  106, 26,  11,  110, 27,  58,  77,  107, 92,  68, 95,  83,  82,
         84,  60,  86,  37,  34,  46,  90,  99,  79,  12,  41, 81,  31,  29,
         4,   91,  45,  65,  105, 103, 67,  102, 71,  78,  40, 39,  74,  111,
         14,  69,  114, 115, 49,  6,   9,   17}));
    S.add_generator(
        Perm({20,  95,  91,  53, 40,  21, 32,  118, 109, 2,   41,  74,  72, 102,
              115, 65,  3,   66, 43,  92, 7,   84,  1,   82,  104, 101, 24, 77,
              97,  57,  71,  35, 50,  56, 60,  42,  80,  85,  58,  22,  34, 88,
              117, 90,  49,  47, 112, 73, 10,  17,  111, 83,  33,  78,  61, 13,
              70,  28,  38,  94, 4,   69, 99,  87,  9,   14,  44,  54,  5,  67,
              52,  62,  116, 96, 51,  79, 105, 27,  16,  107, 108, 26,  19, 11,
              68,  103, 12,  63, 48,  18, 89,  64,  23,  55,  76,  39,  45, 29,
              37,  30,  113, 8,  93,  98, 81,  59,  36,  110, 106, 25,  75, 6,
              46,  114, 119, 15, 86,  31, 0,   100}));
    T.add_generator(
        Perm({22,  119, 110, 87, 4,   12,  16,  102, 98,  39, 113, 90,  5,  62,
              104, 116, 6,   93, 103, 48,  112, 38,  0,   28, 53,  94,  30, 51,
              23,  74,  26,  45, 118, 47,  115, 66,  101, 59, 21,  9,   49, 41,
              114, 96,  85,  31, 75,  33,  19,  40,  99,  27, 97,  24,  54, 57,
              56,  55,  89,  37, 108, 79,  13,  73,  65,  64, 35,  67,  69, 68,
              107, 88,  72,  63, 29,  46,  77,  76,  100, 61, 81,  80,  91, 86,
              105, 44,  83,  3,  71,  58,  11,  82,  106, 17, 25,  95,  43, 52,
              8,   50,  78,  36, 7,   18,  14,  84,  92,  70, 60,  109, 2,  111,
              20,  10,  42,  34, 15,  117, 32,  1}));
    T.add_generator(Perm(
        {109, 40,  7,   74,  81,  3,   83,  113, 31, 53,  112, 10, 52,  19,
         32,  94,  107, 20,  13,  111, 88,  0,   9,  16,  41,  97, 80,  99,
         51,  17,  101, 77,  79,  26,  35,  117, 5,  1,   28,  33, 82,  4,
         76,  60,  38,  43,  67,  21,  34,  61,  96, 44,  73,  93, 116, 14,
         69,  86,  58,  25,  106, 49,  108, 66,  84, 65,  46,  63, 6,   92,
         103, 71,  8,   104, 36,  91,  57,  72,  2,  55,  39,  24, 37,  119,
         30,  100, 42,  59,  29,  56,  15,  105, 89, 22,  95,  90, 62,  87,
         98,  27,  115, 64,  75,  54,  12,  102, 45, 110, 50,  47, 23,  18,
         118, 78,  85,  114, 70,  48,  11,  68}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 20160);
    REQUIRE(U.contains(Perm(
        {0,   115, 24,  104, 82,  16, 88,  86,  106, 9,   39,  118, 40,  70,
         91,  77,  5,   111, 28,  57, 53,  87,  74,  58,  2,   112, 109, 72,
         18,  41,  44,  93,  71,  66, 64,  49,  46,  99,  54,  10,  12,  29,
         117, 78,  30,  68,  36,  48, 47,  35,  61,  119, 83,  20,  38,  90,
         75,  19,  23,  107, 63,  50, 103, 60,  34,  67,  33,  65,  45,  114,
         13,  32,  27,  73,  22,  56, 95,  15,  43,  92,  80,  110, 4,   52,
         113, 101, 7,   21,  6,   98, 55,  14,  79,  31,  94,  76,  102, 97,
         89,  37,  100, 85,  96,  62, 3,   108, 8,   59,  105, 26,  81,  17,
         25,  84,  69,  1,   116, 42, 11,  51})));
    REQUIRE(U.contains(
        Perm({0,   15,  83,  22, 70,  30, 48,  105, 54,  9,   81,  49,  69,  4,
              26,  115, 44,  64, 61,  46, 78,  75,  104, 27,  52,  60,  91,  58,
              50,  68,  16,  55, 107, 98, 17,  11,  19,  65,  8,   110, 114, 45,
              119, 20,  5,   29, 57,  6,  88,  118, 18,  42,  2,   43,  106, 93,
              21,  36,  72,  32, 112, 28, 79,  25,  111, 99,  89,  37,  41,  40,
              82,  59,  23,  73, 3,   87, 101, 1,   53,  103, 100, 39,  13,  24,
              102, 76,  108, 56, 47,  33, 31,  109, 62,  90,  97,  85,  84,  94,
              66,  67,  80,  95, 113, 92, 74,  86,  38,  71,  7,   14,  10,  34,
              63,  96,  12,  77, 116, 51, 35,  117})));
    REQUIRE(U.contains(
        Perm({0,   62,  33,  48,  82,  44,  22,  72, 38,  73,  55, 29, 25,  13,
              117, 92,  30,  56,  76,  19,  107, 34, 6,   108, 66, 12, 119, 86,
              95,  11,  16,  81,  78,  2,   21,  68, 46,  96,  8,  90, 112, 118,
              91,  71,  5,   49,  36,  104, 3,   45, 101, 109, 98, 59, 106, 10,
              17,  57,  105, 53,  114, 85,  1,   69, 87,  113, 24, 84, 35,  63,
              70,  43,  7,   9,   88,  111, 18,  79, 32,  77,  80, 31, 4,   89,
              67,  61,  27,  64,  74,  83,  39,  42, 15,  110, 97, 28, 37,  94,
              52,  102, 100, 50,  99,  115, 47,  58, 54,  20,  23, 51, 93,  75,
              40,  65,  60,  103, 116, 14,  41,  26})));
    REQUIRE(U.contains(Perm(
        {0,   96,  62,  16,  21,  97,  48,  43,  31,  119, 117, 66,  60,  63,
         87,  68,  44,  73,  24,  46,  79,  8,   104, 11,  113, 69,  56,  41,
         89,  115, 22,  111, 18,  2,   112, 27,  5,   33,  75,  12,  82,  85,
         9,   28,  19,  99,  74,  94,  88,  71,  107, 40,  98,  105, 14,  39,
         70,  100, 32,  72,  17,  20,  78,  90,  55,  29,  50,  23,  58,  42,
         114, 53,  37,  64,  116, 26,  1,   101, 59,  84,  36,  93,  109, 61,
         86,  95,  92,  91,  80,  65,  54,  13,  83,  25,  30,  45,  103, 6,
         49,  35,  47,  76,  52,  108, 57,  102, 10,  118, 77,  106, 38,  34,
         4,   15,  110, 7,   3,   51,  67,  81})));
    REQUIRE(U.contains(Perm(
        {1,   50,  80, 62,  114, 92,  57,  97,  6,   112, 7,   12,  58,  95,
         119, 54,  32, 106, 109, 56,  78,  15,  72,  44,  113, 25,  23,  29,
         75,  59,  13, 17,  88,  67,  87,  104, 103, 34,  76,  110, 117, 93,
         5,   105, 26, 69,  33,  115, 101, 68,  0,   22,  60,  39,  21,  28,
         64,  8,   11, 27,  84,  45,  116, 2,   19,  35,  4,   46,  98,  61,
         14,  118, 51, 81,  79,  55,  107, 18,  94,  111, 63,  108, 47,  83,
         52,  36,  43, 37,  16,  48,  71,  40,  42,  99,  20,  30,  9,   10,
         49,  41,  24, 89,  102, 85,  65,  86,  31,  38,  73,  77,  53,  74,
         96,  100, 66, 82,  3,   91,  90,  70})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "024",
                          "PSL(3, 4).Sym(3) and PSL(3, 4).2 intersection",
                          "[quick][schreier-sims]") {
    auto              rg = ReportGuard(REPORT);
    SchreierSims<105> S, T, U;
    using Perm = SchreierSims<105>::element_type;
    S.add_generator(
        Perm({0,   32, 24, 16, 34,  10,  46, 43, 44, 33,  5,  93, 49, 13, 101,
              71,  3,  28, 20, 21,  18,  19, 67, 52, 2,   89, 42, 41, 17, 53,
              30,  36, 1,  9,  4,   63,  31, 77, 38, 76,  47, 27, 26, 7,  8,
              66,  6,  40, 72, 12,  59,  73, 23, 29, 90,  97, 78, 74, 82, 50,
              84,  85, 81, 35, 83,  69,  45, 22, 86, 65,  98, 15, 48, 51, 57,
              100, 39, 37, 56, 104, 103, 62, 58, 64, 60,  61, 68, 87, 91, 25,
              54,  88, 96, 11, 95,  94,  92, 55, 70, 102, 75, 14, 99, 80, 79}));
    S.add_generator(
        Perm({54, 18, 98, 67, 42,  22,  24, 35, 55, 8,  5,  104, 16,  15, 103,
              79, 99, 71, 63, 100, 34,  97, 10, 81, 94, 95, 32,  39,  65, 77,
              92, 90, 49, 31, 45,  58,  40, 2,  28, 48, 64, 30,  101, 3,  0,
              20, 78, 14, 27, 26,  102, 75, 73, 59, 44, 9,  52,  70,  7,  76,
              72, 91, 17, 1,  36,  38,  19, 43, 46, 89, 82, 62,  87,  56, 23,
              84, 53, 93, 68, 13,  11,  74, 57, 69, 51, 21, 50,  60,  25, 83,
              33, 96, 41, 29, 6,   88,  61, 85, 37, 12, 66, 4,   86,  47, 80}));
    T.add_generator(
        Perm({56, 87,  10, 6,  59, 95,  20, 2,  96, 17, 80,  85, 47,  22, 104,
              4,  99,  90, 18, 84, 61,  34, 79, 28, 25, 50,  97, 49,  83, 78,
              1,  48,  76, 69, 26, 58,  3,  63, 62, 27, 86,  44, 77,  40, 82,
              29, 73,  41, 94, 51, 102, 65, 11, 52, 81, 19,  67, 55,  33, 12,
              57, 30,  88, 98, 68, 13,  9,  21, 37, 14, 0,   91, 53,  89, 45,
              31, 35,  54, 75, 39, 66,  42, 15, 64, 46, 43,  72, 36,  71, 60,
              7,  103, 93, 24, 74, 8,   5,  70, 23, 38, 101, 92, 100, 16, 32}));
    T.add_generator(
        Perm({31, 49, 50,  8,  45, 99, 51,  5,  40,  54, 22,  75, 85, 66, 19,
              87, 21, 89,  84, 9,  44, 42,  38, 59,  24, 73,  63, 30, 71, 25,
              28, 14, 100, 3,  64, 16, 58,  90, 79,  56, 7,   12, 69, 47, 101,
              46, 98, 18,  29, 72, 11, 91,  13, 37,  86, 68,  88, 36, 65, 53,
              77, 34, 20,  6,  35, 95, 103, 43, 81,  61, 78,  27, 92, 62, 2,
              70, 74, 15,  76, 1,  32, 23,  0,  104, 80, 102, 82, 52, 41, 57,
              55, 83, 10,  17, 26, 93, 39,  33, 4,   97, 67,  48, 96, 60, 94}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 20160);
    REQUIRE(U.contains(Perm(
        {0,  1,   17, 47, 4,  5,  41, 21,  18,  9,   10, 51, 76, 38, 77,
         59, 40,  2,  8,  43, 44, 7,  94,  25,  28,  23, 90, 46, 24, 91,
         87, 65,  32, 33, 34, 81, 69, 101, 13,  49,  16, 6,  54, 19, 20,
         78, 27,  3,  57, 39, 71, 11, 89,  88,  42,  92, 66, 48, 82, 15,
         64, 104, 63, 62, 60, 31, 56, 95,  100, 36,  80, 50, 74, 93, 72,
         86, 12,  14, 45, 85, 70, 35, 58,  84,  83,  79, 75, 30, 53, 52,
         26, 29,  55, 73, 22, 67, 97, 96,  103, 102, 68, 37, 99, 98, 61})));
    REQUIRE(U.contains(Perm(
        {0,   1,  6,  7,  4,  5,   2,   3,  54, 9,  10, 73, 63, 87,  45,
         48,  43, 41, 42, 40, 26,  47,  67, 29, 46, 91, 20, 28, 27,  23,
         38,  97, 32, 33, 34, 49,  55,  56, 30, 81, 19, 17, 18, 16,  90,
         14,  24, 21, 15, 35, 74,  93,  53, 52, 8,  36, 37, 59, 102, 57,
         104, 64, 76, 12, 61, 96,  101, 22, 70, 92, 68, 72, 71, 11,  50,
         103, 62, 78, 77, 84, 100, 39,  99, 85, 79, 83, 98, 13, 89,  88,
         44,  25, 69, 51, 95, 94,  65,  31, 86, 82, 80, 66, 58, 75,  60})));
    REQUIRE(U.contains(Perm(
        {0,  1,  42, 7,  4,  34,  18, 3,   41, 9,  10, 85,  92,  14, 13,
         39, 53, 54, 6,  89, 90,  47, 56,  23, 28, 25, 44,  46,  24, 29,
         78, 80, 33, 32, 5,  57,  62, 67,  77, 15, 88, 8,   2,   52, 26,
         87, 27, 21, 81, 59, 98,  79, 43,  16, 17, 76, 22,  35,  60, 49,
         58, 99, 36, 69, 82, 70,  94, 37,  96, 63, 65, 103, 75,  83, 86,
         72, 55, 38, 30, 51, 31,  48, 64,  73, 93, 11, 74,  45,  40, 19,
         20, 91, 12, 84, 66, 101, 68, 100, 50, 61, 97, 95,  104, 71, 102})));
    REQUIRE(U.contains(Perm(
        {0,   1,   88, 47, 4,  33,  89, 21, 43, 9,  10, 57, 101, 71, 103,
         93,  54,  53, 19, 18, 26,  7,  36, 23, 27, 25, 20, 24,  46, 29,
         74,  102, 34, 5,  32, 85,  22, 76, 50, 84, 42, 52, 40,  8,  90,
         75,  28,  3,  51, 83, 38,  48, 41, 17, 16, 67, 62, 11,  97, 73,
         100, 70,  56, 66, 68, 99,  63, 55, 64, 94, 61, 13, 87,  59, 30,
         45,  37,  98, 86, 81, 104, 79, 96, 49, 39, 35, 78, 72,  2,  6,
         44,  91,  95, 15, 69, 92,  82, 58, 77, 65, 60, 12, 31,  14, 80})));
    REQUIRE(U.contains(Perm(
        {0,  9,  54,  20, 4,  33,  18,  90, 16, 10,  1,  60, 73, 78,  75,
         31, 17, 8,   43, 2,  23,  44,  79, 3,  24,  7,  91, 28, 46,  21,
         45, 92, 32,  34, 5,  102, 59,  39, 77, 99,  41, 42, 40, 6,   29,
         98, 27, 26,  96, 82, 72,  104, 52, 88, 19,  48, 35, 65, 101, 97,
         76, 12, 93,  51, 62, 69,  49,  83, 95, 57,  22, 71, 74, 61,  50,
         87, 11, 103, 86, 70, 67,  58,  66, 80, 100, 68, 13, 14, 89,  53,
         25, 47, 15,  64, 84, 85,  55,  36, 30, 37,  94, 81, 56, 38,  63})));
    REQUIRE(U.contains(Perm(
        {0,  4,   54,  26, 1,  34, 8,   20,  41, 10, 9,  75, 99, 81,  57,
         45, 89,  42,  6,  53, 21, 44,  80,  24, 25, 28, 47, 29, 23,  46,
         49, 56,  33,  32, 5,  13, 64,  96,  35, 30, 52, 18, 2,  88,  7,
         59, 91,  90,  14, 87, 84, 86,  16,  43, 17, 61, 65, 77, 63,  78,
         36, 92,  58,  82, 69, 66, 31,  100, 67, 60, 22, 83, 85, 103, 79,
         51, 102, 48,  15, 72, 94, 38,  62,  50, 71, 74, 11, 39, 19,  40,
         3,  27,  104, 98, 70, 68, 101, 37,  73, 76, 95, 97, 12, 93,  55})));
    REQUIRE(U.contains(Perm(
        {1,   4,  15,  7,  0,  25, 84,  47,  83, 9,  10, 13, 67, 88,  16,
         30,  51, 39,  73, 48, 44, 21,  56,  24, 34, 28, 20, 32, 5,   46,
         2,   80, 91,  29, 23, 75, 12,  63,  40, 45, 79, 93, 49, 81,  26,
         17,  33, 3,   86, 87, 8,  14,  57,  85, 59, 95, 94, 74, 96,  78,
         100, 60, 55,  76, 97, 31, 66,  36,  99, 92, 70, 41, 43, 103, 52,
         89,  37, 53,  54, 38, 65, 72,  68,  50, 98, 77, 19, 42, 11,  35,
         90,  27, 101, 71, 22, 62, 104, 102, 6,  82, 61, 69, 64, 18,  58})));
    REQUIRE(U.contains(Perm(
        {2,   17,  84, 37, 6,  70,  93,  57, 58, 41,  34, 28, 19, 65, 55,
         69,  36,  23, 21, 67, 5,   100, 53, 56, 38,  59, 0,  68, 48, 80,
         86,  62,  30, 66, 15, 103, 16,  42, 63, 95,  90, 39, 96, 12, 32,
         72,  101, 13, 61, 73, 79,  82,  24, 45, 104, 89, 1,  78, 64, 25,
         27,  11,  54, 52, 92, 77,  40,  43, 74, 10,  99, 46, 22, 49, 60,
         102, 4,   47, 81, 97, 91,  7,   35, 83, 26,  50, 44, 29, 71, 98,
         33,  87,  8,  76, 18, 9,   3,   85, 14, 20,  94, 88, 75, 51, 31})));
  }

  LIBSEMIGROUPS_TEST_CASE("SchreierSimsHelper",
                          "025",
                          "3^4:5:4 and D_16:4 intersection",
                          "[quick][schreier-sims]") {
    auto             rg = ReportGuard(REPORT);
    SchreierSims<81> S, T, U;
    using Perm = SchreierSims<81>::element_type;
    S.add_generator(Perm(
        {0,  9,  18, 80, 62, 71, 40, 49, 31, 3,  12, 21, 74, 56, 65, 43, 52,
         34, 6,  15, 24, 77, 59, 68, 37, 46, 28, 27, 36, 45, 26, 8,  17, 67,
         76, 58, 30, 39, 48, 20, 2,  11, 70, 79, 61, 33, 42, 51, 23, 5,  14,
         64, 73, 55, 54, 63, 72, 53, 35, 44, 13, 22, 4,  57, 66, 75, 47, 29,
         38, 16, 25, 7,  60, 69, 78, 50, 32, 41, 10, 19, 1}));
    S.add_generator(Perm(
        {0,  3,  6,  80, 74, 77, 40, 43, 37, 1,  4,  7,  78, 72, 75, 41, 44,
         38, 2,  5,  8,  79, 73, 76, 39, 42, 36, 9,  12, 15, 62, 56, 59, 49,
         52, 46, 10, 13, 16, 60, 54, 57, 50, 53, 47, 11, 14, 17, 61, 55, 58,
         48, 51, 45, 18, 21, 24, 71, 65, 68, 31, 34, 28, 19, 22, 25, 69, 63,
         66, 32, 35, 29, 20, 23, 26, 70, 64, 67, 30, 33, 27}));
    S.add_generator(Perm(
        {1,  2,  0,  4,  5,  3,  7,  8,  6,  10, 11, 9,  13, 14, 12, 16, 17,
         15, 19, 20, 18, 22, 23, 21, 25, 26, 24, 28, 29, 27, 31, 32, 30, 34,
         35, 33, 37, 38, 36, 40, 41, 39, 43, 44, 42, 46, 47, 45, 49, 50, 48,
         52, 53, 51, 55, 56, 54, 58, 59, 57, 61, 62, 60, 64, 65, 63, 67, 68,
         66, 70, 71, 69, 73, 74, 72, 76, 77, 75, 79, 80, 78}));
    T.add_generator(Perm(
        {0,  6,  3,  5,  2,  8,  7,  4,  1,  36, 42, 39, 41, 38, 44, 43, 40,
         37, 72, 78, 75, 77, 74, 80, 79, 76, 73, 9,  15, 12, 14, 11, 17, 16,
         13, 10, 45, 51, 48, 50, 47, 53, 52, 49, 46, 54, 60, 57, 59, 56, 62,
         61, 58, 55, 18, 24, 21, 23, 20, 26, 25, 22, 19, 27, 33, 30, 32, 29,
         35, 34, 31, 28, 63, 69, 66, 68, 65, 71, 70, 67, 64}));
    T.add_generator(Perm(
        {0,  2,  1,  6,  8,  7,  3,  5,  4,  9,  11, 10, 15, 17, 16, 12, 14,
         13, 18, 20, 19, 24, 26, 25, 21, 23, 22, 27, 29, 28, 33, 35, 34, 30,
         32, 31, 36, 38, 37, 42, 44, 43, 39, 41, 40, 45, 47, 46, 51, 53, 52,
         48, 50, 49, 54, 56, 55, 60, 62, 61, 57, 59, 58, 63, 65, 64, 69, 71,
         70, 66, 68, 67, 72, 74, 73, 78, 80, 79, 75, 77, 76}));
    T.add_generator(Perm(
        {0,  9,  18, 27, 36, 45, 54, 63, 72, 1,  10, 19, 28, 37, 46, 55, 64,
         73, 2,  11, 20, 29, 38, 47, 56, 65, 74, 3,  12, 21, 30, 39, 48, 57,
         66, 75, 4,  13, 22, 31, 40, 49, 58, 67, 76, 5,  14, 23, 32, 41, 50,
         59, 68, 77, 6,  15, 24, 33, 42, 51, 60, 69, 78, 7,  16, 25, 34, 43,
         52, 61, 70, 79, 8,  17, 26, 35, 44, 53, 62, 71, 80}));
    T.add_generator(Perm(
        {0,  2,  1,  4,  3,  5,  8,  7,  6,  45, 47, 46, 49, 48, 50, 53, 52,
         51, 63, 65, 64, 67, 66, 68, 71, 70, 69, 27, 29, 28, 31, 30, 32, 35,
         34, 33, 72, 74, 73, 76, 75, 77, 80, 79, 78, 9,  11, 10, 13, 12, 14,
         17, 16, 15, 54, 56, 55, 58, 57, 59, 62, 61, 60, 18, 20, 19, 22, 21,
         23, 26, 25, 24, 36, 38, 37, 40, 39, 41, 44, 43, 42}));
    T.add_generator(Perm(
        {1,  2,  0,  4,  5,  3,  7,  8,  6,  10, 11, 9,  13, 14, 12, 16, 17,
         15, 19, 20, 18, 22, 23, 21, 25, 26, 24, 28, 29, 27, 31, 32, 30, 34,
         35, 33, 37, 38, 36, 40, 41, 39, 43, 44, 42, 46, 47, 45, 49, 50, 48,
         52, 53, 51, 55, 56, 54, 58, 59, 57, 61, 62, 60, 64, 65, 63, 67, 68,
         66, 70, 71, 69, 73, 74, 72, 76, 77, 75, 79, 80, 78}));
    schreier_sims_helper::intersection(U, S, T);
    REQUIRE(U.size() == 162);
    REQUIRE(U.contains(Perm(
        {0,  9,  18, 27, 36, 45, 54, 63, 72, 1,  10, 19, 28, 37, 46, 55, 64,
         73, 2,  11, 20, 29, 38, 47, 56, 65, 74, 3,  12, 21, 30, 39, 48, 57,
         66, 75, 4,  13, 22, 31, 40, 49, 58, 67, 76, 5,  14, 23, 32, 41, 50,
         59, 68, 77, 6,  15, 24, 33, 42, 51, 60, 69, 78, 7,  16, 25, 34, 43,
         52, 61, 70, 79, 8,  17, 26, 35, 44, 53, 62, 71, 80})));
    REQUIRE(U.contains(Perm(
        {1,  2,  0,  4,  5,  3,  7,  8,  6,  10, 11, 9,  13, 14, 12, 16, 17,
         15, 19, 20, 18, 22, 23, 21, 25, 26, 24, 28, 29, 27, 31, 32, 30, 34,
         35, 33, 37, 38, 36, 40, 41, 39, 43, 44, 42, 46, 47, 45, 49, 50, 48,
         52, 53, 51, 55, 56, 54, 58, 59, 57, 61, 62, 60, 64, 65, 63, 67, 68,
         66, 70, 71, 69, 73, 74, 72, 76, 77, 75, 79, 80, 78})));
    REQUIRE(U.contains(Perm(
        {3,  4,  5,  6,  7,  8,  0,  1,  2,  12, 13, 14, 15, 16, 17, 9,  10,
         11, 21, 22, 23, 24, 25, 26, 18, 19, 20, 30, 31, 32, 33, 34, 35, 27,
         28, 29, 39, 40, 41, 42, 43, 44, 36, 37, 38, 48, 49, 50, 51, 52, 53,
         45, 46, 47, 57, 58, 59, 60, 61, 62, 54, 55, 56, 66, 67, 68, 69, 70,
         71, 63, 64, 65, 75, 76, 77, 78, 79, 80, 72, 73, 74})));
  }

}  // namespace libsemigroups
