// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/ChargedFinalState.hh"

namespace Rivet {


  /// @brief SFM charged multiplicities in NSD and inelastic minbias events
  class SFM_1984_I196601 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(SFM_1984_I196601);


    /// @name Analysis methods
    /// @{

    void init() {
      // Projections
      //
      declare(ChargedFinalState(Cuts::absrap<5 && Cuts::pT>250*MeV && Cuts::pT<3*GeV), "FS");

      // Histograms
      size_t ih = 1;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal, 1e-2))  _sqs = en;
        book(_h[en+"inel"], 1, 1, ih);
        book(_h[en+"nsd"],  2, 1, ih);
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    // Analyse each event
    void analyze(const Event& event) {
      const ChargedFinalState& fs = apply<ChargedFinalState>(event, "FS");

      // Trigger
      if (fs.particles().size() <1 ) vetoEvent;

      // Event classification:
      int n_left(0), n_right(0), n_large_x(0);
      for (const Particle& p : fs.particles()) {
        // Calculate the particles' Feynman x
        const double x_feyn = 2.0 * fabs(p.pz())/sqrtS();
        if (x_feyn > 0.8 ) n_large_x += 1;

        // Pseudorapidity
        const double eta = p.eta();
        if (eta > 0.0) n_right += 1;
        else if (eta < 0.0) n_left += 1;
      }
      MSG_DEBUG("N_left: " << n_left << ", "
                << "N_right: " << n_right << ", "
                << "N_large_x: " << n_large_x);


      // Single diffractive: either one large x particle or 0 particles in the one hemisphere but more than 7 in the other hemisphere
      bool isDiffractive = (n_large_x == 1) ||
                           ( ((n_left==0) && (fs.particles().size() < 7)) ||
                             ((n_right==0) && (fs.particles().size() < 7)) );


      _h[_sqs+"inel"]->fill(fs.particles().size());
      if (!isDiffractive) _h[_sqs+"nsd"]->fill(fs.particles().size());
    }


    void finalize() {
      normalize(_h);
    }

    /// @}


  private:


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    /// @}

  };



  RIVET_DECLARE_ALIASED_PLUGIN(SFM_1984_I196601, SFM_1984_S1178091);

}
