// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief e+ e- -> pi0 pi0 J/psi
  class BESIII_2020_I1793431 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2020_I1793431);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // set the PDG code
      _pid = getOption<double>("PID", 9030443);
      // projections
      declare(FinalState(), "FS");

      // histos
      size_t ih = 1;
      for (double eval : { 4.226, 4.236, 4.244, 4.258 }) {

        const string en = toString(round(eval/MeV));
        if (isCompatibleWithSqrtS(eval))  _sqs = en;

        book(_h[en+"_1"], 2, ih, 1);
        book(_h[en+"_2"], 2, ih, 2);
        ++ih;
      }

      book(_sigma[0], 1, 1, 1);
      const auto& ref = refData<YODA::BinnedEstimate<string>>(1,1,2);
      book(_sigma[1],"TMP/numer", ref);
      book(_sigma[2],"TMP/denom", ref);
      book(_sigma[3], 3, 1, 1);

      for (size_t ix=0; ix<3; ++ix) {
        for (const string& en : _sigma[ix].binning().edges<0>()) {
          const double end = stod(en);
          if (isCompatibleWithSqrtS(end)) {
            _edges[ix] = en; break;
          }
        }
      }
      raiseBeamErrorIf(_edges[0].empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      Particles fs = apply<FinalState>(event, "FS").particles();
      Particles Jpsi, other;
      for (const Particle& p : fs) {
        Particle parent=p;
        while (!parent.parents().empty()) {
          parent=parent.parents()[0];
          if (parent.abspid()==PID::JPSI) break;
        }
        if (parent.abspid()!=PID::JPSI) {
          other.push_back(p);
          continue;
        }
        bool found=false;
        for (const auto& psi : Jpsi) {
          // J/psi already in list
          if (fuzzyEquals(psi.momentum(),parent.momentum())) {
            found=true;
            break;
          }
        }
        if (!found) Jpsi.push_back(parent);
      }
      if (Jpsi.size()!=1 || other.size()!=2) vetoEvent;
      if (other[0].pid()==PID::PI0 && other[1].pid()==PID::PI0) {
        _sigma[0]->fill(_edges[0]);
        if (!_edges[1].empty()) _sigma[1]->fill(_edges[1]);
        if (Jpsi[0].parents()[0].pid()==_pid && !_edges[2].empty()) _sigma[3]->fill(_edges[2]);
        if (_sqs != "") {
          for (size_t iy=0; iy<2; ++iy) {
            _h[_sqs+"_1"]->fill((Jpsi[0].mom()+other[iy].mom()).mass());
          }
          _h[_sqs+"_2"]->fill((other[0].mom()+other[1].mom()).mass());
        }
      }
      else if (other[0].pid()==-other[1].pid() &&
               other[0].abspid()==PID::PIPLUS &&
               !_edges[1].empty()) {
        _sigma[2]->fill(_edges[1]);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {

      normalize(_h, 1.0, false);
      scale(_sigma, crossSection()/ sumOfWeights() /picobarn);
      BinnedEstimatePtr<string> tmp;
      book(tmp,1,1,2);
      divide(_sigma[1], _sigma[2], tmp);
    }

    /// @}


    /// @name Histograms
    /// @{
    int _pid;
    map<string,Histo1DPtr> _h;
    BinnedHistoPtr<string> _sigma[4];
    string _edges[3], _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESIII_2020_I1793431);

}
