//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// G4ParticleChangeForDecay
//
// Class description:
//
// Concrete class for ParticleChange which has functionality for G4Decay.
//
// This class contains the results after invocation of the decay process.
// This includes secondary particles generated by the interaction.

// Author: Hisaya Kurashige, 23 March 1998  
// --------------------------------------------------------------------
#ifndef G4ParticleChangeForDecay_hh
#define G4ParticleChangeForDecay_hh 1

#include "globals.hh"
#include "G4ios.hh"
#include "G4ThreeVector.hh"
#include "G4VParticleChange.hh"

class G4DynamicParticle;

class G4ParticleChangeForDecay final : public G4VParticleChange
{
  public:

    G4ParticleChangeForDecay();

    ~G4ParticleChangeForDecay() override = default;

    G4ParticleChangeForDecay(const G4ParticleChangeForDecay& right) = delete;
    G4ParticleChangeForDecay& operator=(const G4ParticleChangeForDecay& right) = delete;

  // --- the following methods are for updating G4Step -----
  // Return the pointer to the G4Step after updating the step information
  // by using final state information of the track given by a physics process
  // !!! No effect for  AlongSteyp

    G4Step* UpdateStepForAtRest(G4Step* Step) final;
    G4Step* UpdateStepForPostStep(G4Step* Step) final;

    void Initialize(const G4Track&) final;
      // Initialize all properties by using G4Track information

    void ProposeGlobalTime(G4double t);
    void ProposeLocalTime(G4double t);
      // Get/Propose the final global/local time
      // NOTE: DO NOT INVOKE both methods in a step
      //       Each method affects both local and global time

    G4double GetGlobalTime(G4double timeDelay = 0.0) const;
    G4double GetLocalTime(G4double timeDelay = 0.0) const;
      // Convert the time delay to the glocbal/local time.
      // Can get  the final global/local time without argument

    const G4ThreeVector* GetPolarization() const;
    void ProposePolarization(G4double Px, G4double Py, G4double Pz);
    void ProposePolarization(const G4ThreeVector& finalPoralization);
      // Get/Propose the final Polarization vector

  // --- Dump and debug methods ---

    void DumpInfo() const final;

    G4bool CheckIt(const G4Track&) final;

  private:

    G4double theGlobalTime0 = 0.0;
      // The global time at Initial
    G4double theLocalTime0 = 0.0;
      // The local time at Initial

    G4double theTimeChange = 0.0;
      // The change of local time of a given particle

    G4ThreeVector thePolarizationChange;
      // The changed (final) polarization of a given track
};

// ----------------------
// Inline methods
// ----------------------

inline
void G4ParticleChangeForDecay::ProposeGlobalTime(G4double t)
{
  theTimeChange = (t - theGlobalTime0) + theLocalTime0;
}

inline
G4double G4ParticleChangeForDecay::GetGlobalTime(G4double timeDelay) const
{
  //  Convert the time delay to the global time.
  return theGlobalTime0 + (theTimeChange - theLocalTime0) + timeDelay;
}

inline
void G4ParticleChangeForDecay::ProposeLocalTime(G4double t)
{
  theTimeChange = t;
}

inline
G4double G4ParticleChangeForDecay::GetLocalTime(G4double timeDelay) const
{
  //  Convert the time delay to the local time.
  return theTimeChange + timeDelay;
}

inline
const G4ThreeVector* G4ParticleChangeForDecay::GetPolarization() const
{
  return &thePolarizationChange;
}

inline
void G4ParticleChangeForDecay::ProposePolarization(
  const G4ThreeVector& finalPoralization)
{
  thePolarizationChange = finalPoralization;
}

inline
void G4ParticleChangeForDecay::ProposePolarization(G4double Px,
                                                   G4double Py,
                                                   G4double Pz)
{
  thePolarizationChange.setX(Px);
  thePolarizationChange.setY(Py);
  thePolarizationChange.setZ(Pz);
}

#endif
