/*
    SPDX-FileCopyrightText: 2012 Alex Richardson <alex.richardson@gmx.de>

    SPDX-License-Identifier: LGPL-2.1-only OR LGPL-3.0-only OR LicenseRef-KDE-Accepted-LGPL
*/

#include "pointerscriptclass.hpp"

// lib
#include <pointerdatainformation.hpp>
#include <parserutils.hpp>
#include <scriptvalueconverter.hpp>
// Std
#include <utility>

PointerScriptClass::PointerScriptClass(QScriptEngine* engine, ScriptHandlerInfo* handlerInfo)
    : PrimitiveScriptClass(engine, handlerInfo, 4)
    , s_type(engine->toStringHandle(ParserStrings::PROPERTY_TYPE()))
    , s_target(engine->toStringHandle(ParserStrings::PROPERTY_TARGET()))
    , s_scale(engine->toStringHandle(ParserStrings::PROPERTY_SCALE()))
    , s_interpretFunc(engine->toStringHandle(ParserStrings::PROPERTY_INTERPRET_FUNC()))
{
    appendProperty(s_type, QScriptValue::Undeletable);
    appendProperty(s_target, QScriptValue::Undeletable);
    appendProperty(s_scale, QScriptValue::Undeletable);
    appendProperty(s_interpretFunc, QScriptValue::Undeletable);
}

PointerScriptClass::~PointerScriptClass() = default;

QScriptValue PointerScriptClass::additionalProperty(const DataInformation* data, const QScriptString& name,
                                                    uint id)
{
    Q_ASSERT(data->isPointer());
    if (name == s_type) {
        return PrimitiveType::standardTypeName(data->asPointer()->pointerType()->type());
    }
    if (name == s_target) {
        return data->asPointer()->pointerTarget()->toScriptValue(engine(), handlerInfo());
    }
    if (name == s_scale) {
        return engine()->toScriptValue(data->asPointer()->pointerScale());
    }
    if (name == s_interpretFunc) {
        return data->asPointer()->interpreterFunction();
    }
    return PrimitiveScriptClass::additionalProperty(data, name, id);
}

bool PointerScriptClass::queryAdditionalProperty(const DataInformation* data, const QScriptString& name,
                                                 QScriptClass::QueryFlags* flags, uint* id)
{
    if (name == s_type || name == s_target || name == s_scale || name == s_interpretFunc) {
        *flags = QScriptClass::HandlesReadAccess | QScriptClass::HandlesWriteAccess;
        return true;
    }
    return PrimitiveScriptClass::queryAdditionalProperty(data, name, flags, id);
}

bool PointerScriptClass::setAdditionalProperty(DataInformation* data, const QScriptString& name, uint id,
                                               const QScriptValue& value)
{
    if (name == s_type) {
        std::unique_ptr<DataInformation> newType = ScriptValueConverter::convert(value, QStringLiteral("(pointer value)"),
                                                                                 data->logger(), data);
        if (!newType) {
            data->logError() << "Could not set new pointer type.";
        } else {
            std::ignore = data->asPointer()->setPointerType(std::move(newType));
        }

        return true;
    }
    if (name == s_target) {
        std::unique_ptr<DataInformation> newTarget = ScriptValueConverter::convert(value, QStringLiteral("(pointer value)"),
                                                                                   data->logger(), data);
        if (!newTarget) {
            data->logError() << "Could not set new pointer target.";
        } else {
            data->asPointer()->setPointerTarget(std::move(newTarget));
        }
        return true;
    }
    if (name == s_scale) {
        if (!value.isNumber()) {
            data->logError() << "Could not set new pointer scale.";
        } else {
            data->asPointer()->setPointerScale(value.toInteger());
        }
        return true;
    }
    if (name == s_interpretFunc) {
        if (!value.isFunction()) {
            data->logError() << "Could not set new pointer interpreter function.";
        } else {
            data->asPointer()->setInterpreterFunction(value);
        }
        return true;
    }
    return PrimitiveScriptClass::setAdditionalProperty(data, name, id, value);
}
