"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateCustomError = generateCustomError;
exports.getConnectorsFromResponses = getConnectorsFromResponses;
exports.getModelsFromResponses = getModelsFromResponses;
exports.getResourcesCreatedFromResponse = getResourcesCreatedFromResponse;
exports.getWorkflowStateFromResponse = getWorkflowStateFromResponse;
exports.getWorkflowsFromResponses = getWorkflowsFromResponses;
exports.isIgnorableError = isIgnorableError;
exports.toWorkflowObj = toWorkflowObj;
var _common = require("../../common");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

// OSD does not provide an interface for this response, but this is following the suggested
// implementations. To prevent typescript complaining, leaving as loosely-typed 'any'
function generateCustomError(res, err) {
  var _err$body;
  if (isDatasourceError(err)) {
    return res.customError({
      statusCode: 404,
      body: {
        message: 'Data source not found'
      }
    });
  }
  return res.customError({
    statusCode: err.statusCode || 500,
    body: {
      message: err.message,
      attributes: {
        error: ((_err$body = err.body) === null || _err$body === void 0 ? void 0 : _err$body.error) || err.message
      }
    }
  });
}
function isDatasourceError(err) {
  if (err.message !== undefined && typeof err.message === 'string') {
    return err.message.includes(_common.INVALID_DATASOURCE_MSG);
  }
}

// Helper fn to filter out backend errors that we don't want to propagate on the frontend.
function isIgnorableError(error) {
  var _error$body, _error$body2, _error$body3;
  return ((_error$body = error.body) === null || _error$body === void 0 || (_error$body = _error$body.error) === null || _error$body === void 0 ? void 0 : _error$body.type) === _common.INDEX_NOT_FOUND_EXCEPTION || ((_error$body2 = error.body) === null || _error$body2 === void 0 || (_error$body2 = _error$body2.error) === null || _error$body2 === void 0 || (_error$body2 = _error$body2.caused_by) === null || _error$body2 === void 0 ? void 0 : _error$body2.type) === _common.INDEX_NOT_FOUND_EXCEPTION || ((_error$body3 = error.body) === null || _error$body3 === void 0 ? void 0 : _error$body3.error) === _common.NO_MODIFICATIONS_FOUND_TEXT;
}

// Convert backend workflow into frontend workflow obj
function toWorkflowObj(hitSource, id) {
  return {
    id,
    name: hitSource.name,
    use_case: hitSource.use_case,
    description: hitSource.description || '',
    version: hitSource.version,
    workflows: hitSource.workflows,
    ui_metadata: hitSource.ui_metadata,
    lastUpdated: hitSource.last_updated_time,
    lastLaunched: hitSource.last_provisioned_time
  };
}

// TODO: can remove or simplify if we can fetch all data from a single API call. Tracking issue:
// https://github.com/opensearch-project/flow-framework/issues/171
// Current implementation combines 2 search responses to create a single set of workflows with
// static information + state information
function getWorkflowsFromResponses(workflowHits, workflowStateHits) {
  const workflowDict = {};
  workflowHits.forEach(workflowHit => {
    var _workflowStateHit$_so, _workflowStateHit$_so2, _workflowStateHit$_so3;
    const hitSource = workflowHit._source;
    workflowDict[workflowHit._id] = toWorkflowObj(hitSource, workflowHit._id);
    const workflowStateHit = workflowStateHits.find(workflowStateHit => workflowStateHit._id === workflowHit._id);
    const workflowState = getWorkflowStateFromResponse(workflowStateHit === null || workflowStateHit === void 0 || (_workflowStateHit$_so = workflowStateHit._source) === null || _workflowStateHit$_so === void 0 ? void 0 : _workflowStateHit$_so.state);
    const workflowError = workflowStateHit === null || workflowStateHit === void 0 || (_workflowStateHit$_so2 = workflowStateHit._source) === null || _workflowStateHit$_so2 === void 0 ? void 0 : _workflowStateHit$_so2.error;
    const workflowResourcesCreated = getResourcesCreatedFromResponse(workflowStateHit === null || workflowStateHit === void 0 || (_workflowStateHit$_so3 = workflowStateHit._source) === null || _workflowStateHit$_so3 === void 0 ? void 0 : _workflowStateHit$_so3.resources_created);
    workflowDict[workflowHit._id] = {
      ...workflowDict[workflowHit._id],
      // @ts-ignore
      state: workflowState,
      error: workflowError,
      resourcesCreated: workflowResourcesCreated
    };
  });
  return workflowDict;
}
function getModelsFromResponses(modelHits) {
  const modelDict = {};
  modelHits.forEach(modelHit => {
    // search model API returns hits for each deployed model chunk. ignore these hits
    if (modelHit._source.chunk_number === undefined) {
      var _modelHit$_source, _modelHit$_source2, _modelHit$_source3, _modelHit$_source4, _modelHit$_source5, _modelHit$_source6;
      const modelId = modelHit._id;

      // the persisted model interface (if available) is a mix of an obj and string.
      // We parse the string values for input/output to have a complete
      // end-to-end JSONSchema obj
      let indexedModelInterface = modelHit._source.interface;
      let modelInterface = undefined;
      if (indexedModelInterface !== undefined) {
        let parsedInput = undefined;
        let parsedOutput = undefined;
        try {
          parsedInput = JSON.parse(indexedModelInterface.input);
        } catch {}
        try {
          parsedOutput = JSON.parse(indexedModelInterface.output);
        } catch {}
        modelInterface = {
          input: parsedInput,
          output: parsedOutput
        };
      }

      // in case of schema changes from ML plugin, this may crash. That is ok, as the error
      // produced will help expose the root cause
      modelDict[modelId] = {
        id: modelId,
        name: (_modelHit$_source = modelHit._source) === null || _modelHit$_source === void 0 ? void 0 : _modelHit$_source.name,
        // @ts-ignore
        algorithm: _common.MODEL_ALGORITHM[(_modelHit$_source2 = modelHit._source) === null || _modelHit$_source2 === void 0 ? void 0 : _modelHit$_source2.algorithm],
        // @ts-ignore
        state: _common.MODEL_STATE[(_modelHit$_source3 = modelHit._source) === null || _modelHit$_source3 === void 0 ? void 0 : _modelHit$_source3.model_state],
        modelConfig: {
          modelType: (_modelHit$_source4 = modelHit._source) === null || _modelHit$_source4 === void 0 || (_modelHit$_source4 = _modelHit$_source4.model_config) === null || _modelHit$_source4 === void 0 ? void 0 : _modelHit$_source4.model_type,
          embeddingDimension: (_modelHit$_source5 = modelHit._source) === null || _modelHit$_source5 === void 0 || (_modelHit$_source5 = _modelHit$_source5.model_config) === null || _modelHit$_source5 === void 0 ? void 0 : _modelHit$_source5.embedding_dimension
        },
        interface: modelInterface,
        connectorId: (_modelHit$_source6 = modelHit._source) === null || _modelHit$_source6 === void 0 ? void 0 : _modelHit$_source6.connector_id
      };
    }
  });
  return modelDict;
}
function getConnectorsFromResponses(modelHits) {
  const connectorDict = {};
  modelHits.forEach(connectorHit => {
    var _connectorHit$_source, _connectorHit$_source2, _connectorHit$_source3;
    const connectorId = connectorHit._id;

    // in case of schema changes from ML plugin, this may crash. That is ok, as the error
    // produced will help expose the root cause
    connectorDict[connectorId] = {
      id: connectorId,
      name: (_connectorHit$_source = connectorHit._source) === null || _connectorHit$_source === void 0 ? void 0 : _connectorHit$_source.name,
      parameters: {
        model: (_connectorHit$_source2 = connectorHit._source) === null || _connectorHit$_source2 === void 0 || (_connectorHit$_source2 = _connectorHit$_source2.parameters) === null || _connectorHit$_source2 === void 0 ? void 0 : _connectorHit$_source2.model,
        dimensions: (_connectorHit$_source3 = connectorHit._source) === null || _connectorHit$_source3 === void 0 ? void 0 : _connectorHit$_source3.parameters.dimensions
      }
    };
  });
  return connectorDict;
}

// Convert the workflow state into a readable/presentable state on frontend
function getWorkflowStateFromResponse(state) {
  const finalState = state || _common.DEFAULT_NEW_WORKFLOW_STATE_TYPE;
  // @ts-ignore
  return _common.WORKFLOW_STATE[finalState];
}

// Convert the workflow resources into a readable/presentable state on frontend
function getResourcesCreatedFromResponse(resourcesCreated) {
  const finalResources = [];
  if (resourcesCreated) {
    resourcesCreated.forEach(backendResource => {
      finalResources.push({
        id: backendResource.resource_id,
        stepType: backendResource.workflow_step_name,
        type:
        // @ts-ignore
        _common.WORKFLOW_RESOURCE_TYPE[
        // the backend persists the types in lowercase. e.g., "pipeline_id"
        backendResource.resource_type.toUpperCase()]
      });
    });
  }
  return finalResources;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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