/*
 * Copyright (c) 1997, 2012, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.sun.xml.internal.ws.api.addressing;

import com.sun.istack.internal.NotNull;

import com.sun.xml.internal.ws.api.WSBinding;
import com.sun.xml.internal.ws.api.WSService;
import com.sun.xml.internal.ws.api.message.Packet;
import com.sun.xml.internal.ws.api.pipe.ClientTubeAssemblerContext;
import com.sun.xml.internal.ws.api.pipe.Fiber;
import com.sun.xml.internal.ws.api.pipe.TransportTubeFactory;
import com.sun.xml.internal.ws.api.pipe.Tube;
import com.sun.xml.internal.ws.api.server.WSEndpoint;
import com.sun.xml.internal.ws.binding.BindingImpl;

/**
 * Delivers response messages targeted at non-anonymous endpoint addresses
 * @since 2.2.6
 */
public class NonAnonymousResponseProcessor {
        private static final NonAnonymousResponseProcessor DEFAULT = new NonAnonymousResponseProcessor();

        public static NonAnonymousResponseProcessor getDefault() {
                return DEFAULT;
        }

        protected NonAnonymousResponseProcessor() {}

    /**
     * Send a response to a non-anonymous address. Also closes the transport back channel
     * of {@link Packet} if it's not closed already.
     *
     * @param packet
     *      The response from our server, which will be delivered to the destination.
     * @return The response packet that should be used to complete the tubeline response processing
     */
        public Packet process(Packet packet) {
        Fiber.CompletionCallback fiberCallback = null;
        Fiber currentFiber = Fiber.getCurrentIfSet();
        if (currentFiber != null) {
            // Link completion of the current fiber to the new fiber that will
            // deliver the async response. This allows access to the response
            // packet that may be generated by sending a new message for the
            // current async response.

                final Fiber.CompletionCallback currentFiberCallback =
                    currentFiber.getCompletionCallback();

                        if (currentFiberCallback != null) {
                          fiberCallback = new Fiber.CompletionCallback() {
                          public void onCompletion(@NotNull Packet response) {
                            currentFiberCallback.onCompletion(response);
                          }

                          public void onCompletion(@NotNull Throwable error) {
                            currentFiberCallback.onCompletion(error);
                          }
                        };
                        currentFiber.setCompletionCallback(null);
                }
        }

        // we need to assemble a pipeline to talk to this endpoint.
                WSEndpoint<?> endpoint = packet.endpoint;
                WSBinding binding = endpoint.getBinding();
        Tube transport = TransportTubeFactory.create(Thread.currentThread().getContextClassLoader(),
            new ClientTubeAssemblerContext(
                        packet.endpointAddress, endpoint.getPort(), (WSService) null,
                        binding, endpoint.getContainer(),
                        ((BindingImpl) binding).createCodec(), null, null));
        Fiber fiber = endpoint.getEngine().createFiber();
        fiber.start(transport, packet, fiberCallback);

        // then we'll proceed the rest like one-way.
        Packet copy = packet.copy(false);
        copy.endpointAddress = null;

        return copy;
        }
}
